# Sistema de Gestão Financeira - Manual Técnico para Desenvolvedores

## Índice

1. [Visão Geral](#visão-geral)
2. [Arquitetura do Sistema](#arquitetura-do-sistema)  
3. [Estrutura do Projeto](#estrutura-do-projeto)
4. [Modelos e Banco de Dados](#modelos-e-banco-de-dados)
5. [Controllers](#controllers)
6. [Sistema de Autenticação](#sistema-de-autenticação)
7. [Rotas e Middleware](#rotas-e-middleware)
8. [Views e Interface](#views-e-interface)
9. [Funcionalidades Principais](#funcionalidades-principais)
10. [Exportação PDF/Excel](#exportação-pdfexcel)
11. [Boas Práticas](#boas-práticas)
12. [Guia de Desenvolvimento](#guia-de-desenvolvimento)

---

## 1. Visão Geral

### Descrição do Sistema
Sistema web desenvolvido em Laravel 10 para gestão financeira de projetos, permitindo controle completo do ciclo de vida de projetos desde submissão até conclusão.

### Stack Tecnológica
- **Framework:** Laravel 10.x
- **PHP:** ^8.1
- **Frontend:** Bootstrap 5, Remix Icons, JavaScript ES6
- **PDF:** DomPDF (barryvdh/laravel-dompdf)
- **Banco de Dados:** MySQL/MariaDB
- **Servidor:** Apache/Nginx

### Funcionalidades Principais
- Gestão de projetos com workflow de aprovação
- Sistema de roles (Admin/Técnico)
- Exportação PDF/Excel
- Dashboard com métricas
- Controle de desembolsos
- Sistema de anexos

---

## 2. Arquitetura do Sistema

### Padrão MVC
O sistema segue rigorosamente o padrão Model-View-Controller do Laravel:

```
app/
├── Models/           # Eloquent Models
├── Http/Controllers/ # Business Logic
├── Http/Middleware/  # Request Filtering
└── Http/Requests/    # Form Validation

resources/views/      # Blade Templates
routes/web.php       # Route Definitions
```

### Design Patterns Implementados

#### 1. Repository Pattern (Implícito)
- Controllers usam Eloquent Models diretamente
- Scopes customizados para queries complexas

#### 2. Factory Pattern
- Database Factories para testing e seeding

#### 3. Observer Pattern
- Eloquent Events para ações automáticas

---

## 3. Estrutura do Projeto

### Diretório Principal
```
FINANCIAL/
├── app/
│   ├── Http/Controllers/
│   ├── Models/
│   └── Jobs/
├── database/
│   ├── migrations/
│   ├── factories/
│   └── seeders/
├── resources/
│   ├── views/
│   │   ├── Admin/
│   │   ├── Clientes/
│   │   └── pdf/
│   └── css/
├── routes/
└── public/
```

### Organização das Views
- **Admin/**: Interface administrativa
- **Clientes/**: Interface do usuário final  
- **pdf/**: Templates para exportação PDF
- **Mail/**: Templates de email

---

## 4. Modelos e Banco de Dados

### Modelo Principal: Projecto

```php
class Projecto extends Model
{
    protected $fillable = [
        'codProjecto', 'nomeProjecto', 'descricaoObjectivo',
        'limiteorcamentalP', 'status', 'admin_responsavel_id'
        // ... outros campos
    ];
    
    // Relacionamentos
    public function tipoProjecto() { return $this->belongsTo(TipoProjecto::class); }
    public function tecnicoProjecto() { return $this->belongsTo(User::class); }
    public function cronogramasFisicos() { return $this->hasMany(CronogramaFisico::class); }
    public function programacoesFinanceiras() { return $this->hasMany(ProgramacaoFinanceira::class); }
}
```

### Status do Projeto
```php
0 => 'Recebido'      // Visível para todos admins
1 => 'Em Análise'    // Apenas admin responsável
2 => 'Aprovado'      // Apenas admin responsável  
3 => 'Rejeitado'     // Apenas admin responsável
4 => 'Em Execução'   // Apenas admin responsável
5 => 'Concluído'     // Apenas admin responsável
```

### Relacionamentos Principais

```php
// Um projeto tem muitos cronogramas
Projecto 1:N CronogramaFisico

// Um projeto tem muitas programações financeiras  
Projecto 1:N ProgramacaoFinanceira

// Um projeto pertence a um tipo
Projecto N:1 TipoProjecto

// Um projeto tem um técnico responsável
Projecto N:1 User (tecnico_projectoFK)

// Um projeto pode ter um admin responsável
Projecto N:1 User (admin_responsavel_id)
```

### Scopes Importantes

```php
// Controle de acesso administrativo
public function scopeVisivelParaAdmin($query, $adminId)
{
    return $query->where(function ($q) use ($adminId) {
        $q->where('status', 0) // Status Recebido: todos veem
            ->orWhere('admin_responsavel_id', $adminId); // Outros: só responsável
    });
}
```

---

## 5. Controllers

### ProjectoController - Funcionalidades Principais

#### Métodos para Usuários (Técnicos)
```php
index()           // Formulário de criação
show()            // Listar projetos do usuário
store()           // Salvar novo projeto
edit($id)         // Formulário de edição
update($id)       // Atualizar projeto
destroy($id)      // Excluir projeto
```

#### Métodos Administrativos
```php
indexAdmin()      // Lista todos os projetos (com filtros)
showAdmin($id)    // Detalhes para admin
atualizarStatus() // Mudança de status
obterDetalhes()   // AJAX para modal de detalhes
```

#### Métodos de Exportação
```php
exportarPDF($id)                    // PDF individual
exportarExcel($id)                  // Excel individual
exportarPDFListagemGeral()         // PDF da listagem
exportarExcelListagemGeral()       // Excel da listagem
visualizarPDFListagemGeral()       // Preview PDF
```

### HomeController - Dashboard

```php
dashboardAdmin()  // Métricas administrativas
index()           // Dashboard padrão
```

### Padrão de Resposta dos Controllers

```php
// Sucesso com redirect
return redirect()->route('route.name')->with('success', 'Mensagem');

// Erro com redirect  
return back()->withErrors(['erro' => 'Mensagem de erro']);

// JSON para AJAX
return response()->json(['status' => 'success', 'data' => $data]);
```

---

## 6. Sistema de Autenticação

### Middleware de Autorização

```php
// Middleware 'admin' - routes/web.php
Route::middleware('admin')->group(function () {
    Route::get('/admin/projetos', [ProjectoController::class, 'indexAdmin']);
    // ... outras rotas administrativas
});
```

### Controle de Acesso por Status

**Regra de Negócio Crítica:**
- Status 0 (Recebido): Qualquer admin pode visualizar
- Status 1-5: Apenas o admin que alterou o status pode continuar

```php
// Implementação no modelo
public function podeAlterarStatus($adminId)
{
    if ($this->status == 0) return true;  // Qualquer admin
    return $this->admin_responsavel_id == $adminId;  // Só responsável
}
```

### Tipos de Usuário
- **Admin**: Acesso total ao sistema administrativo
- **Técnico**: Pode criar/editar apenas seus projetos

---

## 7. Rotas e Middleware

### Estrutura das Rotas

```php
// Rotas públicas
Route::get('/', [AcessoController::class, 'index'])->name('login');

// Rotas administrativas protegidas
Route::middleware('admin')->group(function () {
    // Gestão de projetos
    Route::get('/admin/projetos', [ProjectoController::class, 'indexAdmin']);
    
    // Exportações
    Route::get('/admin/projetos-listagem/exportar-pdf', 
        [ProjectoController::class, 'exportarPDFListagemGeral']);
});

// Rotas de usuário autenticado  
Route::middleware('auth')->group(function () {
    Route::get('/projetos/criar', [ProjectoController::class, 'index']);
    Route::post('/projetos', [ProjectoController::class, 'store']);
});
```

### Middleware Personalizado

```php
// app/Http/Middleware/AdminMiddleware.php
public function handle($request, Closure $next)
{
    if (!auth()->user() || !auth()->user()->isAdmin()) {
        return redirect('/login');
    }
    return $next($request);
}
```

---

## 8. Views e Interface

### Sistema de Layout
- **Base:** `app.blade.php` - Layout principal
- **Seções:** Admin, Clientes, PDF, Mail
- **Componentes:** Modals, Cards, Forms reutilizáveis

### Tecnologias Frontend
```html
<!-- Bootstrap 5 -->
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css">

<!-- Remix Icons -->
<link href="https://cdn.jsdelivr.net/npm/remixicon@3.5.0/fonts/remixicon.css">

<!-- Custom CSS -->
<link href="{{ asset('assets/css/app.css') }}">
```

### Padrões de Interface

#### Cards Responsivos
```html
<div class="card">
    <div class="card-header">
        <h5 class="card-title">
            <i class="ri-icon-name me-2"></i>Título
        </h5>
    </div>
    <div class="card-body">
        <!-- Conteúdo -->
    </div>
</div>
```

#### Dropdowns de Ação
```html
<div class="dropdown">
    <button class="btn btn-primary dropdown-toggle" data-bs-toggle="dropdown">
        <i class="ri-settings-line me-1"></i>Ações
    </button>
    <ul class="dropdown-menu">
        <li><a class="dropdown-item" href="#"><i class="ri-eye-line me-2"></i>Visualizar</a></li>
        <li><a class="dropdown-item" href="#"><i class="ri-download-line me-2"></i>Baixar</a></li>
    </ul>
</div>
```

---

## 9. Funcionalidades Principais

### Gestão de Projetos

#### Workflow de Status
1. **Submissão:** Técnico cria projeto (Status: 0 - Recebido)
2. **Análise:** Admin altera para "Em Análise" (torna-se responsável)
3. **Decisão:** Admin aprova/rejeita
4. **Execução:** Se aprovado, pode ir para "Em Execução"
5. **Conclusão:** Finalização do projeto

#### Filtros Implementados
```php
// No controller
if ($request->filled('status')) {
    $query->where('status', $request->status);
}

if ($request->filled('search')) {
    $query->where(function ($q) use ($request) {
        $q->where('nomeProjecto', 'like', '%'.$request->search.'%')
          ->orWhere('codProjecto', 'like', '%'.$request->search.'%');
    });
}
```

### Sistema de Anexos
Campos suportados:
- `Url_contratoP` - Contrato do Projeto
- `ambito` - Âmbito do Projeto  
- `impactoSocial` - Estudo de Impacto Social
- `tecnicoEngenharia` - Parecer Técnico
- `analiseEfectividade` - Análise de Efetividade
- `viabilidadeEconomica` - Viabilidade Econômica
- `impactoAmbiental` - Impacto Ambiental
- `mitigacaoImpactos` - Mitigação de Impactos

### Dashboard com Métricas
```php
// Cálculo de métricas
$projetosRecebidos = Projecto::where('status', 0)
    ->whereMonth('created_at', now()->month)->count();
    
$prazoFinanciamento = Projecto::where('status', 2)
    ->where('created_at', '>=', now()->subDays(30))->count();
```

---

## 10. Exportação PDF/Excel

### Templates PDF
Localizados em `resources/views/pdf/`:
- `ficha-projeto-final.blade.php` - PDF individual
- `gestao-projetos-listagem.blade.php` - Listagem geral
- `desembolso-geral.blade.php` - Relatório de desembolsos

### Configuração DomPDF
```php
$pdf = Pdf::loadView('pdf.template-name', $data);
$pdf->setPaper('A4', 'landscape'); // ou 'portrait'
return $pdf->download('filename.pdf'); // ou ->stream()
```

### Exportação Excel (CSV)
```php
$headers = [
    'Content-Type' => 'text/csv; charset=UTF-8',
    'Content-Disposition' => 'attachment; filename="' . $filename . '"',
];

$callback = function() use ($data) {
    $file = fopen('php://output', 'w');
    fprintf($file, chr(0xEF).chr(0xBB).chr(0xBF)); // BOM UTF-8
    fputcsv($file, $headers, ';');
    // ... dados
    fclose($file);
};

return response()->stream($callback, 200, $headers);
```

---

## 11. Boas Práticas Implementadas

### 1. Segurança
- Middleware de autenticação e autorização
- Validação de dados com Form Requests
- Proteção contra SQL Injection (Eloquent ORM)
- CSRF Protection nativo do Laravel

### 2. Performance
- Eager Loading para relacionamentos (`with()`)
- Paginação em listagens extensas
- Índices de banco de dados otimizados
- Cache de queries quando necessário

### 3. Manutenibilidade  
- Código organizado seguindo PSR-4
- Comentários em funções críticas
- Nomes descritivos de métodos e variáveis
- Separação clara de responsabilidades

### 4. UX/UI
- Interface responsiva (Bootstrap 5)
- Feedback visual para ações do usuário
- Loading states e confirmações
- Dropdowns organizados para ações

### 5. Tratamento de Erros
```php
try {
    DB::transaction(function () use ($dados) {
        // Operações críticas
    });
    return redirect()->back()->with('success', 'Operação realizada');
} catch (\Exception $e) {
    return redirect()->back()->withErrors(['erro' => 'Falha na operação']);
}
```

---

## 12. Guia de Desenvolvimento

### Setup do Ambiente
```bash
# Instalar dependências
composer install
npm install

# Configurar ambiente  
cp .env.example .env
php artisan key:generate

# Executar migrações
php artisan migrate --seed

# Compilar assets
npm run dev
```

### Comandos Úteis
```bash
# Gerar controller
php artisan make:controller NomeController

# Gerar model com migration
php artisan make:model Nome -m

# Executar testes
php artisan test

# Limpar cache
php artisan cache:clear
php artisan config:clear
```

### Estrutura para Novos Módulos
1. Criar Model com relationships
2. Criar Migration com foreign keys
3. Criar Controller com CRUD completo
4. Definir rotas no web.php
5. Criar views seguindo padrão existente
6. Implementar validações
7. Adicionar testes

### Debugging
- Use `dd()` para debug rápido
- `Log::info()` para logs em produção  
- Laravel Telescope para debug avançado
- Artisan Tinker para testes de models

---

## Considerações Finais

Este sistema foi desenvolvido seguindo as melhores práticas do Laravel e design patterns consolidados. A arquitetura é escalável e permite fácil manutenção e extensão de funcionalidades.

Para dúvidas ou contribuições, consulte a documentação oficial do Laravel em https://laravel.com/docs

---

**Documento gerado em:** {{ date('d/m/Y H:i:s') }}
**Versão do Sistema:** Laravel 10.x
**Última atualização:** {{ date('d/m/Y') }}
