<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class SalarioProcessado extends Model
{
    use HasFactory;

    protected $table = 'salarios_processados';

    protected $fillable = [
        'nomeTrabalhador',
        'nbi',
        'email_funcionario',
        'funcao',
        'grupoPessoal',
        'tipoClasse',
        'descricao',
        'salarioBase',
        'abonoFamiliar',
        'subTransporte',
        'subAlimentacao',
        'subFerias',
        'subOutros',
        'desconto',
        'acrescimoExtra',
        'total',
        'status_envio_email',
        'data_envio_email',
        'erro_envio_email',
        'processamento_salarios_id'
    ];

    protected $casts = [
        'salarioBase' => 'decimal:2',
        'abonoFamiliar' => 'decimal:2',
        'subTransporte' => 'decimal:2',
        'subAlimentacao' => 'decimal:2',
        'subFerias' => 'decimal:2',
        'subOutros' => 'decimal:2',
        'desconto' => 'decimal:2',
        'acrescimoExtra' => 'decimal:2',
        'total' => 'decimal:2',
        'data_envio_email' => 'datetime',
    ];

    /**
     * Relacionamento com Processamento de Salário
     */
    public function processamentoSalario()
    {
        return $this->belongsTo(ProcessamentoSalario::class, 'processamento_salarios_id');
    }

    /**
     * Calcula o total bruto (sem descontos)
     */
    public function getTotalBrutoAttribute()
    {
        return $this->salarioBase 
            + $this->abonoFamiliar 
            + $this->subTransporte 
            + $this->subAlimentacao 
            + $this->subFerias 
            + $this->subOutros
            + $this->acrescimoExtra;
    }

    /**
     * Calcula os descontos totais
     */
    public function getTotalDescontosAttribute()
    {
        return $this->desconto;
    }

    /**
     * Scope para buscar por NBI
     */
    public function scopePorNbi($query, $nbi)
    {
        return $query->where('nbi', $nbi);
    }

    /**
     * Scope para buscar por função
     */
    public function scopePorFuncao($query, $funcao)
    {
        return $query->where('funcao', 'like', '%' . $funcao . '%');
    }

    /**
     * Scope para buscar por grupo pessoal
     */
    public function scopePorGrupoPessoal($query, $grupoPessoal)
    {
        return $query->where('grupoPessoal', 'like', '%' . $grupoPessoal . '%');
    }
}
