<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class RelatorioExecucao extends Model
{
    use HasFactory;

    protected $table = 'relatorios_execucao';

    protected $fillable = [
        'projeto_id',
        'usuario_id',
        'descricao_tecnica',
        'descricao_financeira',
        'periodo_inicio',
        'periodo_fim',
        'percentual_fisico',
        'valor_executado',
        'anexos',
        'observacoes',
        'status',
        'motivo_rejeicao',
        'aprovado_por',
        'aprovado_em',
    ];

    protected $casts = [
        'anexos' => 'array',
        'periodo_inicio' => 'date',
        'periodo_fim' => 'date',
        'percentual_fisico' => 'decimal:2',
        'valor_executado' => 'decimal:2',
        'aprovado_em' => 'datetime',
    ];

    /**
     * Relacionamento com projeto
     */
    public function projeto()
    {
        return $this->belongsTo(Projecto::class, 'projeto_id');
    }

    /**
     * Relacionamento com usuário (técnico)
     */
    public function usuario()
    {
        return $this->belongsTo(User::class, 'usuario_id');
    }

    /**
     * Relacionamento com gestor que aprovou
     */
    public function aprovador()
    {
        return $this->belongsTo(User::class, 'aprovado_por');
    }

    /**
     * Scope para filtrar por projeto
     */
    public function scopeByProjeto($query, $projetoId)
    {
        return $query->where('projeto_id', $projetoId);
    }

    /**
     * Scope para filtrar por usuário
     */
    public function scopeByUsuario($query, $usuarioId)
    {
        return $query->where('usuario_id', $usuarioId);
    }

    /**
     * Scope para filtrar por status
     */
    public function scopeByStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * Getter para status formatado
     */
    public function getStatusFormatadoAttribute()
    {
        $status = [
            'pendente' => 'Pendente',
            'aprovado' => 'Aprovado',
            'rejeitado' => 'Rejeitado',
        ];

        return $status[$this->status] ?? 'Desconhecido';
    }

    /**
     * Getter para cor do status
     */
    public function getStatusCorAttribute()
    {
        $cores = [
            'pendente' => 'warning',
            'aprovado' => 'success',
            'rejeitado' => 'danger',
        ];

        return $cores[$this->status] ?? 'secondary';
    }

    /**
     * Getter para período formatado
     */
    public function getPeriodoFormatadoAttribute()
    {
        return $this->periodo_inicio->format('d/m/Y').' a '.$this->periodo_fim->format('d/m/Y');
    }

    /**
     * Getter para valor executado formatado
     */
    public function getValorExecutadoFormatadoAttribute()
    {
        return number_format($this->valor_executado, 2, ',', '.').' Kz';
    }

    /**
     * Getter para percentual físico formatado
     */
    public function getPercentualFisicoFormatadoAttribute()
    {
        return $this->percentual_fisico ? $this->percentual_fisico.'%' : 'N/A';
    }
}
