<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Projecto extends Model
{
    use HasFactory;

    protected $table = 'projectos';

    // Chave primária
    protected $primaryKey = 'id';

    // Indica se a chave primária é um incremento automático
    public $incrementing = true;

    // Tipo de chave primária
    protected $keyType = 'int';

    // Campos da tabela
    protected $fillable = ['codProjecto', 'nomeProjecto', 'descricaoObjectivo', 'limiteorcamentalP', 'beneficiarioDP', 'duracaoP', 'duracaoF', 'justificacao', 'aprovadoPor', 'financeador', 'Url_contratoP', 'ambito', 'mFuncionamento', 'localizacao_projecto', 'impactoSocial', 'tecnicoEngenharia', 'analiseEfectividade', 'viabilidadeEconomica', 'impactoAmbiental', 'tipoProjectFK', 'gestor_projectoFK', 'tecnico_projectoFK', 'empregosDirectos', 'kitsDistribuidos', 'aprovadoPor', 'facturaPath', 'mitigacaoImpactos', 'status', 'admin_responsavel_id', 'data_mudanca_status'];

    public function tipoProjecto()
    {
        return $this->belongsTo(TipoProjecto::class, 'tipoProjectFK');
    }

    public function user()
    {
        return $this->belongsTo(User::class, 'gestor_projectoFK');
    }

    public function tecnicoProjecto()
    {
        return $this->belongsTo(User::class, 'tecnico_projectoFK');
    }

    /**
     * Relacionamento com cronograma físico das atividades
     */
    public function cronogramasFisicos()
    {
        return $this->hasMany(CronogramaFisico::class, 'projeto_id');
    }

    /**
     * Relacionamento com programações financeiras
     */
    public function programacoesFinanceiras()
    {
        return $this->hasMany(ProgramacaoFinanceira::class, 'projeto_id');
    }

    /**
     * Relacionamento com relatórios de execução
     */
    public function relatoriosExecucao()
    {
        return $this->hasMany(RelatorioExecucao::class, 'projeto_id');
    }

    /**
     * Relacionamento com pagamentos do projeto
     */
    public function pagamentos()
    {
        return $this->hasMany(Pagamento::class, 'projecto_id');
    }

    /**
     * Accessor para status formatado
     */
    public function getStatusFormatadoAttribute()
    {
        $status = [
            0 => 'Recebido',
            1 => 'Em Análise',
            2 => 'Aprovado',
            3 => 'Rejeitado',
            4 => 'Em Execução',
            5 => 'Concluído',
            6 => 'Cancelado',
        ];

        return $status[$this->status] ?? 'Desconhecido';
    }

    /**
     * Accessor para cor do status
     */
    public function getStatusCorAttribute()
    {
        $cores = [
            0 => 'warning',    // Recebido
            1 => 'info',       // Em Análise
            2 => 'success',    // Aprovado
            3 => 'danger',     // Rejeitado
            4 => 'primary',    // Em Execução
            5 => 'success',    // Concluído
            6 => 'secondary',   // Cancelado
        ];

        return $cores[$this->status] ?? 'secondary';
    }

    /**
     * Accessor para valor formatado
     */
    public function getLimiteOrcamentalFormatadoAttribute()
    {
        return 'AOA '.number_format($this->limiteorcamentalP, 2, ',', '.');
    }

    /**
     * Relacionamento com iniciativas
     */
    public function iniciativas()
    {
        return $this->hasMany(Iniciativa::class, 'projecto_id');
    }

    public function scopeSearch($query, $search)
    {
        return $query->where('nomeProjecto', 'like', "%{$search}%")
            ->orWhere('codProjecto', 'like', "%{$search}%");
    }

    public function scopeFilterByStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * Scope para filtrar projetos por usuário
     */
    public function scopeByUser($query, $userId)
    {
        return $query->where('gestor_projectoFK', $userId);
    }

    /**
     * Relacionamento com o admin responsável
     */
    public function adminResponsavel()
    {
        return $this->belongsTo(User::class, 'admin_responsavel_id');
    }

    /**
     * Scope para projetos visíveis para administrador
     *
     * REGRA DE NEGÓCIO:
     * - Status 0 (Recebido): Todos os administradores podem visualizar
     *   (pois ainda não há admin responsável atribuído)
     * - Status 1-5: Apenas o admin responsável pode visualizar
     *   (admin_responsavel_id é definido quando o status sai de 0)
     */
    public function scopeVisivelParaAdmin($query, $adminId)
    {
        return $query->where(function ($q) use ($adminId) {
            $q->where('status', 0) // Status Recebido: todos os admins podem ver
                ->orWhere('admin_responsavel_id', $adminId); // Outros status: só o admin responsável
        });
    }

    /**
     * Verifica se o admin pode alterar o status do projeto
     *
     * REGRA DE NEGÓCIO:
     * - Status 0 (Recebido): Qualquer admin pode alterar (e se torna responsável)
     * - Status 1-5: Apenas o admin responsável pode continuar alterando
     */
    public function podeAlterarStatus($adminId)
    {
        // Status 0 (Recebido): qualquer admin pode alterar e se torna responsável
        if ($this->status == 0) {
            return true;
        }

        // Status 1-5: apenas o admin responsável pode alterar
        // return $this->admin_responsavel_id == $adminId;
        return true;
    }

    /**
     * Verifica próximos status válidos baseado no status atual
     */
    public function proximosStatusValidos()
    {
        $transicoes = [
            0 => [1], // Recebido → Em Análise
            1 => [2, 3], // Em Análise → Aprovado ou Rejeitado
            2 => [4], // Aprovado → Em Execução
            3 => [], // Rejeitado (não pode mudar)
            4 => [5], // Em Execução → Concluído
            5 => [], // Concluído (não pode mudar)
        ];

        return $transicoes[$this->status] ?? [];
    }

    /**
     * Calcula o valor total pago para este projeto
     */
    public function valorTotalPago()
    {
        return $this->pagamentos()->sum('valor');
    }
}
