<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class ProcessamentoSalario extends Model
{
    use HasFactory;

    protected $table = 'processamento_salarios';

    protected $fillable = [
        'descricao',
        'dataProcessamento',
        'ano',
        'mes',
        'unidadePagadora',
        'unidadeProcessadora',
        'status'
    ];

    protected $casts = [
        'dataProcessamento' => 'datetime',
    ];

    /**
     * Mutator para definir o status padrão como 'Processando'
     */
    protected static function boot()
    {
        parent::boot();

        static::saving(function ($processamento) {
            if (!$processamento->exists) {
                $processamento->status = $processamento->status ?? 'Processando';
            }
        });
    }

    /**
     * Relacionamento com Salários Processados
     */
    public function salariosProcessados()
    {
        return $this->hasMany(SalarioProcessado::class, 'processamento_salarios_id');
    }

    /**
     * Scope para processamentos por ano
     */
    public function scopePorAno($query, $ano)
    {
        return $query->where('ano', $ano);
    }

    /**
     * Scope para processamentos por mês
     */
    public function scopePorMes($query, $mes)
    {
        return $query->where('mes', $mes);
    }

    /**
     * Scope para processamentos por status
     */
    public function scopePorStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * Retorna o total de salários processados
     */
    public function totalSalarios()
    {
        return $this->salariosProcessados()->sum('total');
    }

    /**
     * Retorna a quantidade de funcionários processados
     */
    public function quantidadeFuncionarios()
    {
        return $this->salariosProcessados()->count();
    }

    /**
     * Retorna nome do mês por extenso
     */
    public function getNomeMesAttribute()
    {
        $meses = [
            1 => 'Janeiro', 2 => 'Fevereiro', 3 => 'Março', 4 => 'Abril',
            5 => 'Maio', 6 => 'Junho', 7 => 'Julho', 8 => 'Agosto',
            9 => 'Setembro', 10 => 'Outubro', 11 => 'Novembro', 12 => 'Dezembro'
        ];
        return $meses[$this->mes] ?? 'N/A';
    }

    /**
     * Retorna período de referência formatado
     */
    public function getPeriodoReferenciaAttribute()
    {
        return $this->nomeMes . '/' . $this->ano;
    }
}
