<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Funcionario extends Model
{
    use HasFactory;

    protected $table = 'funcionarios';

    // Campos da tabela
    protected $fillable = [
        'nomeCompleto',
        'numeroDocumento',
        'tipoDocumento',
        'checkDocumento',
        'telefone',
        'checkTelefone',
        'email',
        'numeroColaborador',
        'estado',
        'id_departamento',
        'id_funcao',
        'id_grupo_pessoal',
        // Coordenadas Bancárias
        'banco',
        'iban',
        'nib',
        'swift_bic',
        'observacoes_bancarias'
    ];

    /**
     * Mutator para definir o estado padrão como 'Ativo'
     */
    protected static function boot()
    {
        parent::boot();

        static::saving(function ($funcionario) {
            if (!$funcionario->exists) {
                $funcionario->estado = $funcionario->estado ?? 'Ativo';
                $funcionario->checkDocumento = $funcionario->checkDocumento ?? '0';
                $funcionario->checkTelefone = $funcionario->checkTelefone ?? '0';
            }
        });
    }

    /**
     * Accessor para formatar o estado
     */
    public function getEstadoAttribute($value)
    {
        return ucfirst($value);
    }

    /**
     * Accessor para formatar checkDocumento
     */
    public function getCheckDocumentoAttribute($value)
    {
        return $value == '1' ? 'Verificado' : 'Não Verificado';
    }

    /**
     * Accessor para formatar checkTelefone
     */
    public function getCheckTelefoneAttribute($value)
    {
        return $value == '1' ? 'Verificado' : 'Não Verificado';
    }

    /**
     * Relacionamento com Departamento
     */
    public function departamento()
    {
        return $this->belongsTo(Departamento::class, 'id_departamento');
    }

    /**
     * Relacionamento com Função
     */
    public function funcao()
    {
        return $this->belongsTo(Funcoes::class, 'id_funcao');
    }

    /**
     * Relacionamento com Grupo Pessoal
     */
    public function grupoPessoal()
    {
        return $this->belongsTo(GrupoPessoais::class, 'id_grupo_pessoal');
    }

    /**
     * Scope para funcionários ativos
     */
    public function scopeAtivos($query)
    {
        return $query->where('estado', 'Ativo');
    }

    /**
     * Scope para funcionários inativos
     */
    public function scopeInativos($query)
    {
        return $query->where('estado', 'Inativo');
    }

    /**
     * Scope para buscar por nome completo
     */
    public function scopePorNome($query, $nome)
    {
        return $query->where('nomeCompleto', 'like', '%' . $nome . '%');
    }

    /**
     * Scope para buscar por departamento
     */
    public function scopePorDepartamento($query, $departamentoId)
    {
        return $query->where('id_departamento', $departamentoId);
    }

    /**
     * Scope para buscar por função
     */
    public function scopePorFuncao($query, $funcaoId)
    {
        return $query->where('id_funcao', $funcaoId);
    }

    /**
     * Relacionamento com Remunerações (muitos-para-muitos) via pivot funcionario_remuneracao
     */
    public function remuneracoes()
    {
        return $this->belongsToMany(Remuneracao::class, 'funcionario_remuneracao', 'funcionario_id', 'remuneracao_id')
            ->withPivot(['valor', 'observacao', 'status'])
            ->withTimestamps();
    }
}
