<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class CoordenadasBancarias extends Model
{
    use HasFactory;

    protected $table = 'coordenadas_bancarias';

    // Campos da tabela
    protected $fillable = [
        'nomeBanco',
        'siglaBanco',
        'ContaBancaria',
        'Iban',
        'id_titular',
        'estado'
    ];

    /**
     * Mutator para definir o estado padrão como 'Ativo'
     */
    protected static function boot()
    {
        parent::boot();

        static::saving(function ($coordenadasBancarias) {
            if (!$coordenadasBancarias->exists) {
                $coordenadasBancarias->estado = $coordenadasBancarias->estado ?? 'Ativo';
            }
        });
    }

    /**
     * Accessor para formatar o estado
     */
    public function getEstadoAttribute($value)
    {
        return ucfirst($value);
    }

    /**
     * Relacionamento com Funcionário (titular da conta)
     */
    public function funcionario()
    {
        return $this->belongsTo(Funcionario::class, 'id_titular');
    }

    /**
     * Scope para coordenadas bancárias ativas
     */
    public function scopeAtivas($query)
    {
        return $query->where('estado', 'Ativo');
    }

    /**
     * Scope para coordenadas bancárias inativas
     */
    public function scopeInativas($query)
    {
        return $query->where('estado', 'Inativo');
    }

    /**
     * Scope para buscar por nome do banco
     */
    public function scopePorBanco($query, $nomeBanco)
    {
        return $query->where('nomeBanco', 'like', '%' . $nomeBanco . '%');
    }

    /**
     * Scope para buscar por sigla do banco
     */
    public function scopePorSiglaBanco($query, $siglaBanco)
    {
        return $query->where('siglaBanco', 'like', '%' . $siglaBanco . '%');
    }

    /**
     * Scope para buscar por titular (funcionário)
     */
    public function scopePorTitular($query, $titularId)
    {
        return $query->where('id_titular', $titularId);
    }
}
