<?php

namespace App\Http\Controllers;

use App\Jobs\EnviarEmailNovoUsuarioJob;
use App\Models\Entidade;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;
use Exception;

class UserController extends Controller
{
    /**
     * Mostra a página de utilizadores do Sistema/Clientes e usuário
     */
    public function index()
    {
        $utilizadores = User::orderByDesc('id')->paginate(10);
        $entidades = Entidade::all();
        return view('Administracao.Utilizador', [
            'utilizadores' => $utilizadores,
            'entidades' => $entidades
        ]);
    }

    /**
     * Mostra o formulário para criar um novo utilizador
     */
    public function create()
    {
        $entidades = Entidade::where('id', '=', 1)->get();
        return view('Administracao.CriarUtilizador', ['entidades' => $entidades]);
    }

    /**
     * Cadastrar utilizador/Promotor ou Administrador
     */
    public function store(Request $request)
    {
        // Validar formulário
        $validator = Validator::make($request->all(), [
            'nome' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users,email',
            'tel' => 'required|string|max:20',
            'id_entidade' => 'required|exists:entidades,id',
            'tipo' => 'required|in:1,2', // 1-Admin, 2-Cliente/Promotor
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }
          
        $utilizador = new User();
        
        // Gerar senha aleatória para o novo utilizador
        $password = rand(100000, 999999); // Senha de 6 dígitos para maior segurança

        $utilizador->name = $request->nome;
        $utilizador->email = $request->email;
        $utilizador->password = bcrypt($password);
        $utilizador->email_verified_at = now();
        $utilizador->remember_token = Str::random(10); 
        $utilizador->telefone = $request->tel;
        $utilizador->id_entidade = $request->id_entidade;

        // Definir tipo de usuário
        if ($request->tipo == "1") { // Administrador
            $utilizador->admin = "1";
            $utilizador->client = "0";
            $tipoUsuario = "Gestor";
            $redirecionamento = 'users.index';
        } else { // Cliente/Promotor
            $utilizador->admin = "0";
            $utilizador->client = "1";
            $tipoUsuario = "Promotor";
            $redirecionamento = 'Listar-EntidadePromotora';
        }
             
        try {
            // Salvar o utilizador na base de dados
            $utilizador->save();
            
            // Adicionar o envio de email à fila de espera
            \App\Jobs\EnviarEmailNovoUsuarioJob::dispatch(
                $request->nome, 
                $request->email, 
                $password, 
                $tipoUsuario
            );
            
            return redirect()->route($redirecionamento)
                ->with('success', $tipoUsuario . ' cadastrado com sucesso! Foi enviado um email com as credenciais.');
            
        } catch (Exception $th) {
            return redirect()->back()
                ->with('error', 'Houve um erro ao cadastrar o usuário: ' . $th->getMessage())
                ->withInput();
        }
    }
    
    /**
     * Exibe os detalhes de um utilizador específico
     */
    public function show($id)
    {
        $utilizador = User::findOrFail($id);
        return view('Administracao.DetalhesUtilizador', [
            'utilizador' => $utilizador
        ]);
    }
    
    /**
     * Mostra o formulário para editar um utilizador
     */
    public function edit($id)
    {
        $utilizador = User::findOrFail($id);
        $entidades = Entidade::all();
        
        return view('Administracao.EditarUtilizador', [
            'utilizador' => $utilizador,
            'entidades' => $entidades
        ]);
    }
    
    /**
     * Atualiza um utilizador específico
     */
    public function update(Request $request, $id)
    {
        $utilizador = User::findOrFail($id);
        
        // Validar formulário
        $validator = Validator::make($request->all(), [
            'nome' => 'required|string|max:255',
            'email' => [
                'required',
                'email',
                Rule::unique('users', 'email')->ignore($id)
            ],
            'tel' => 'required|string|max:20',
            'id_entidade' => 'required|exists:entidades,id',
            'tipo' => 'required|in:1,2', // 1-Admin, 2-Cliente/Promotor
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }
        
        $utilizador->name = $request->nome;
        $utilizador->email = $request->email;
        $utilizador->telefone = $request->tel;
        $utilizador->id_entidade = $request->id_entidade;
        
        // Actualizar senha apenas se fornecida
        if ($request->filled('password')) {
            $utilizador->password = Hash::make($request->password);
        }
        
        // Definir tipo de usuário
        if ($request->tipo == "1") {
            $utilizador->admin = "1";
            $utilizador->client = "0";
        } else {
            $utilizador->admin = "0";
            $utilizador->client = "1";
        }
        
        try {
            $utilizador->save();
            return redirect()->route('users.index')
                ->with('success', 'Usuário atualizado com sucesso!');
        } catch (Exception $th) {
            return redirect()->back()
                ->with('error', 'Houve um erro ao atualizar o usuário: ' . $th->getMessage())
                ->withInput();
        }
    }

    /**
     * Mostra a página de perfil do utilizador autenticado.
     */
    public function showProfile()
    {
       
        $utilizador = auth()->user();

       // dd($utilizador); // Linha de depuração
        
        return view('Clientes.PerfilUtilizador', ['utilizador' => $utilizador]);
    }

    /**
     * Atualiza os dados do perfil do utilizador autenticado.
     */
    public function updateProfile(Request $request)
    {
        $utilizador = auth()->user();

        $rules = [
            'nome' => 'required|string|max:255',
        ];

        if ($request->filled('password')) {
            $rules['password'] = 'required|string|min:6|confirmed';
        }

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return redirect()->route('cliente.perfil.show')
                ->withErrors($validator)
                ->withInput();
        }

        $utilizador->name = $request->nome;

        if ($request->filled('password')) {
            $utilizador->password = Hash::make($request->password);
        }

        try {
            $utilizador->save();
            return redirect()->route('cliente.perfil.show')
                ->with('success', 'Perfil atualizado com sucesso!');
        } catch (Exception $th) {
            return redirect()->route('cliente.perfil.show')
                ->with('error', 'Houve um erro ao atualizar o perfil: ' . $th->getMessage());
        }
    }
    
    /**
     * Remove um utilizador específico
     */
    public function destroy($id)
    {
        try {
            $utilizador = User::findOrFail($id);
            $utilizador->delete();
            
            return redirect()->route('users.index')
                ->with('success', 'Usuário excluído com sucesso!');
        } catch (Exception $th) {
            return redirect()->route('users.index')
                ->with('error', 'Houve um erro ao excluir o usuário: ' . $th->getMessage());
        }
    }
    
    /**
     * Redefine a senha de um utilizador
     */
    public function resetPassword($id)
    {
        try {
            $utilizador = User::findOrFail($id);
            $novasenha = rand(10000, 99999);
            
            $utilizador->password = Hash::make($novasenha);
            $utilizador->save();

            if ($utilizador->admin == "1") { // Administrador
                $tipoUsuario = "Gestor";
            } else { // Cliente/Promotor
                $tipoUsuario = "Promotor";
            }
            
            // Aqui você pode adicionar o envio de email com a nova senha
            $this->eenviarEmail($utilizador->name, $novasenha, $utilizador->email, $tipoUsuario);
            
            return redirect()->route('users.index')
                ->with('success', 'Senha redefinida com sucesso! Foi enviado um email com a nova senha.');
        } catch (Exception $th) {
            return redirect()->route('users.index')
                ->with('error', 'Houve um erro ao redefinir a senha: ' . $th->getMessage());
        }
    }
    
    /**
     * Método para enviar email (implementação pendente)
    */
    public function eenviarEmail($nome, $senha, $email, $tipoUsuario)
    {
       // Adicionar o envio de email à fila de espera
            \App\Jobs\EnviarEmailNovoUsuarioJob::dispatch(
                $nome, 
                $email, 
                $senha, 
                $tipoUsuario
            );
        return true;
    } 
}
