<?php

namespace App\Http\Controllers;

use App\Models\TipoProjecto;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class TipoProjectoController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        // Parâmetros de filtro
        $descricao = $request->input('descricao');
        $status = $request->input('status');
        
        // Query base
        $query = TipoProjecto::query();
        
        // Aplicar filtros se fornecidos
        if ($descricao) {
            $query->where('descricao', 'like', '%' . $descricao . '%');
        }
        
        if ($status !== null && $status !== '') {
            $query->where('status', $status);
        }
        
        // Ordenar e paginar resultados
        $tipos = $query->orderBy('id', 'desc')
                      ->paginate(10)
                      ->withQueryString();
        
        return view('Administracao.TipoProjecto', compact('tipos'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('Administracao.CriarTipoProjecto');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // Validação
        $validator = Validator::make($request->all(), [
            'tipo_projecto' => 'required|string|max:60|unique:tipos_projeto,descricao',
        ], [
            'tipo_projecto.required' => 'A descrição é obrigatória',
            'tipo_projecto.max' => 'A descrição não pode ter mais de 60 caracteres',
            'tipo_projecto.unique' => 'Este tipo de projeto já existe'
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // Criar o tipo de projeto
        TipoProjecto::create([
            'descricao' => $request->tipo_projecto,
            'status' => 1 // Ativo por padrão
        ]);

        return redirect()->route('tipo-projecto')
            ->with('success', 'Tipo de projeto criado com sucesso!');
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        $tipo = TipoProjecto::findOrFail($id);
        return view('Administracao.DetalheTipoProjecto', compact('tipo'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        $tipo = TipoProjecto::findOrFail($id);
        return view('Administracao.EditarTipoProjecto', compact('tipo'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $tipo = TipoProjecto::findOrFail($id);

        // Validação
        $validator = Validator::make($request->all(), [
            'descricao' => 'required|string|max:60|unique:tipos_projeto,descricao,' . $id,
        ], [
            'descricao.required' => 'A descrição é obrigatória',
            'descricao.max' => 'A descrição não pode ter mais de 60 caracteres',
            'descricao.unique' => 'Este tipo de projeto já existe'
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // Atualizar o tipo de projeto
        $tipo->descricao = $request->descricao;
        $tipo->save();

        return redirect()->route('tipo-projecto')
            ->with('success', 'Tipo de projeto atualizado com sucesso!');
    }

    /**
     * Toggle the status of the specified resource.
     */
    public function toggleStatus($id)
    {
        $tipo = TipoProjecto::findOrFail($id);
        
        // Alternar status (1 para ativo, 0 para inativo)
        $tipo->status = ($tipo->status == 1) ? 0 : 1;
        $tipo->save();

        $statusText = $tipo->status == 1 ? 'ativado' : 'desativado';
        
        return redirect()->route('tipo-projecto')
            ->with('success', "Tipo de projeto $statusText com sucesso!");
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $tipo = TipoProjecto::findOrFail($id);
        
        // Verificar se o tipo está sendo usado em algum projeto
        // Aqui você pode adicionar lógica para verificar relações, se necessário
        
        $tipo->delete();

        return redirect()->route('tipo-projecto')
            ->with('success', 'Tipo de projeto excluído com sucesso!');
    }
}
