<?php

namespace App\Http\Controllers;

use App\Models\Projecto;
use App\Models\RelatorioExecucao;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class RelatorioExecucaoController extends Controller
{
    /**
     * Listar relatórios de um projeto
     */
    public function index($projetoId)
    {
        $projeto = Projecto::with(['relatoriosExecucao.usuario', 'relatoriosExecucao.aprovador'])
            ->findOrFail($projetoId);

        // Verificar se o usuário pode acessar este projeto
        if ($projeto->tecnico_projectoFK !== Auth::id() && $projeto->gestor_projectoFK !== Auth::id()) {
            abort(403, 'Acesso negado');
        }

        $relatorios = $projeto->relatoriosExecucao()
            ->with(['usuario', 'aprovador'])
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return view('Clientes.RelatoriosExecucao.index', compact('projeto', 'relatorios'));
    }

    /**
     * Exibir formulário de criação de relatório
     */
    public function create($projetoId)
    {
        $projeto = Projecto::findOrFail($projetoId);

        // Verificar se o usuário é o técnico do projeto e o projeto está em execução
        if ($projeto->tecnico_projectoFK !== Auth::id() || $projeto->status !== 4) {
            abort(403, 'Acesso negado. O projeto deve estar em execução e você deve ser o técnico responsável.');
        }

        return view('Clientes.RelatoriosExecucao.create', compact('projeto'));
    }

    /**
     * Armazenar novo relatório
     */
    public function store(Request $request, $projetoId)
    {
        $projeto = Projecto::findOrFail($projetoId);

        // Verificar permissões
        if (! Auth::check()) {
            return response()->json(['success' => false, 'message' => 'Usuário não autenticado. Faça login novamente.'], 401);
        }

        if ($projeto->tecnico_projectoFK !== Auth::id()) {
            return response()->json(['success' => false, 'message' => 'Apenas o técnico responsável pode criar relatórios para este projeto.'], 403);
        }

        if ($projeto->status !== '4') {
            return response()->json(['success' => false, 'message' => 'Relatórios só podem ser criados para projetos em execução.'.$projeto->status], 403);
        }

        // Validação
        $validator = Validator::make($request->all(), [
            'descricao_tecnica' => 'required|string|min:50|max:2000',
            'descricao_financeira' => 'required|string|min:50|max:2000',
            'periodo_inicio' => 'required|date|before_or_equal:periodo_fim|after_or_equal:'.$projeto->duracaoP,
            'periodo_fim' => 'required|date|after_or_equal:periodo_inicio|before_or_equal:today',
            'percentual_fisico' => 'nullable|numeric|min:0|max:100',
            'valor_executado' => 'nullable|numeric|min:0|max:'.($projeto->valorProjecto ?? 999999999),
            'observacoes' => 'nullable|string|max:1000',
            'anexos.*' => 'nullable|file|mimes:pdf,doc,docx,xls,xlsx,jpg,jpeg,png,gif|max:10240', // 10MB max
        ], [
            'descricao_tecnica.required' => 'A descrição técnica é obrigatória',
            'descricao_tecnica.min' => 'A descrição técnica deve ter pelo menos 50 caracteres',
            'descricao_tecnica.max' => 'A descrição técnica não pode exceder 2000 caracteres',
            'descricao_financeira.required' => 'A descrição financeira é obrigatória',
            'descricao_financeira.min' => 'A descrição financeira deve ter pelo menos 50 caracteres',
            'descricao_financeira.max' => 'A descrição financeira não pode exceder 2000 caracteres',
            'periodo_inicio.required' => 'A data de início é obrigatória',
            'periodo_inicio.after_or_equal' => 'A data de início não pode ser anterior ao início do projeto',
            'periodo_fim.required' => 'A data de fim é obrigatória',
            'periodo_inicio.before_or_equal' => 'A data de início deve ser anterior ou igual à data de fim',
            'periodo_fim.after_or_equal' => 'A data de fim deve ser posterior ou igual à data de início',
            'periodo_fim.before_or_equal' => 'A data de fim não pode ser superior à data atual',
            'percentual_fisico.max' => 'O percentual físico não pode ser superior a 100%',
            'percentual_fisico.min' => 'O percentual físico deve ser positivo',
            'valor_executado.min' => 'O valor executado deve ser positivo',
            'valor_executado.max' => 'O valor executado não pode exceder o valor total do projeto',
            'anexos.*.mimes' => 'Os anexos devem ser arquivos PDF, DOC, DOCX, XLS, XLSX, JPG, JPEG, PNG ou GIF',
            'anexos.*.max' => 'Cada anexo deve ter no máximo 10MB',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Dados inválidos',
                'errors' => $validator->errors(),
            ], 422);
        }

        // Verificar se o usuário está autenticado
        if (! Auth::check()) {
            return response()->json([
                'success' => false,
                'message' => 'Usuário não autenticado',
            ], 401);
        }

        try {
            // Processar anexos
            $anexosPaths = [];
            if ($request->hasFile('anexos')) {
                foreach ($request->file('anexos') as $arquivo) {
                    $nomeOriginal = $arquivo->getClientOriginalName();
                    $extensao = $arquivo->getClientOriginalExtension();
                    $nomeArquivo = time().'_'.uniqid().'.'.$extensao;

                    $path = $arquivo->storeAs('relatorios/anexos', $nomeArquivo, 'public');

                    $anexosPaths[] = [
                        'path' => $path,
                        'nome_original' => $nomeOriginal,
                        'tamanho' => $arquivo->getSize(),
                    ];
                }
            }

            // Criar relatório
            $relatorio = RelatorioExecucao::create([
                'projeto_id' => $projetoId,
                'usuario_id' => Auth::id(),
                'descricao_tecnica' => $request->descricao_tecnica,
                'descricao_financeira' => $request->descricao_financeira,
                'periodo_inicio' => $request->periodo_inicio,
                'periodo_fim' => $request->periodo_fim,
                'percentual_fisico' => $request->percentual_fisico,
                'valor_executado' => $request->valor_executado,
                'anexos' => $anexosPaths,
                'observacoes' => $request->observacoes,
                'status' => 'pendente',
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Relatório de execução criado com sucesso!',
                'redirect' => route('projetos.relatorios.index', $projetoId),
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro ao criar relatório: '.$e->getMessage(),
            ], 500);
        }
    }

    /**
     * Exibir relatório específico
     */
    public function show($projetoId, $relatorioId)
    {
        $projeto = Projecto::findOrFail($projetoId);
        $relatorio = RelatorioExecucao::with(['usuario', 'aprovador'])
            ->where('projeto_id', $projetoId)
            ->findOrFail($relatorioId);

        // Verificar se o usuário pode acessar
        if ($projeto->tecnico_projectoFK !== Auth::id() && $projeto->gestor_projectoFK !== Auth::id()) {
            abort(403, 'Acesso negado');
        }

        return view('Clientes.RelatoriosExecucao.show', compact('projeto', 'relatorio'));
    }

    /**
     * Editar relatório
     */
    public function edit($projetoId, $relatorioId)
    {
        $projeto = Projecto::findOrFail($projetoId);
        $relatorio = RelatorioExecucao::where('projeto_id', $projetoId)->findOrFail($relatorioId);

        // Verificar permissões (só o técnico pode editar e só se estiver pendente)
        if ($projeto->tecnico_projectoFK !== Auth::id() || $relatorio->status !== 'pendente') {
            return response()->json(['success' => false, 'message' => 'Acesso negado'], 403);
        }

        return response()->json([
            'success' => true,
            'relatorio' => [
                'id' => $relatorio->id,
                'periodo_inicio' => $relatorio->periodo_inicio->format('Y-m-d'),
                'periodo_fim' => $relatorio->periodo_fim->format('Y-m-d'),
                'descricao_tecnica' => $relatorio->descricao_tecnica,
                'descricao_financeira' => $relatorio->descricao_financeira,
                'percentual_fisico' => $relatorio->percentual_fisico,
                'valor_executado' => $relatorio->valor_executado,
                'observacoes' => $relatorio->observacoes,
                'anexos' => $relatorio->anexos ?? [],
            ],
        ]);
    }

    /**
     * Atualizar relatório
     */
    public function update(Request $request, $projetoId, $relatorioId)
    {
        $projeto = Projecto::findOrFail($projetoId);
        $relatorio = RelatorioExecucao::where('projeto_id', $projetoId)->findOrFail($relatorioId);

        // Verificar permissões
        if ($projeto->tecnico_projectoFK !== Auth::id() || $relatorio->status !== 'pendente') {
            return response()->json(['success' => false, 'message' => 'Acesso negado'], 403);
        }

        // Usar a mesma validação do store
        $validator = Validator::make($request->all(), [
            'descricao_tecnica' => 'required|string|min:50|max:2000',
            'descricao_financeira' => 'required|string|min:50|max:2000',
            'periodo_inicio' => 'required|date|before_or_equal:periodo_fim|after_or_equal:'.$projeto->duracaoP,
            'periodo_fim' => 'required|date|after_or_equal:periodo_inicio|before_or_equal:today',
            'percentual_fisico' => 'nullable|numeric|min:0|max:100',
            'valor_executado' => 'nullable|numeric|min:0|max:'.($projeto->valorProjecto ?? 999999999),
            'observacoes' => 'nullable|string|max:1000',
            'anexos.*' => 'nullable|file|mimes:pdf,doc,docx,xls,xlsx,jpg,jpeg,png,gif|max:10240',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Dados inválidos',
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            // Processar novos anexos
            $anexosExistentes = $relatorio->anexos ?? [];
            $novosAnexos = [];

            if ($request->hasFile('anexos')) {
                foreach ($request->file('anexos') as $arquivo) {
                    $nomeOriginal = $arquivo->getClientOriginalName();
                    $extensao = $arquivo->getClientOriginalExtension();
                    $nomeArquivo = time().'_'.uniqid().'.'.$extensao;

                    $path = $arquivo->storeAs('relatorios/anexos', $nomeArquivo, 'public');

                    $novosAnexos[] = [
                        'path' => $path,
                        'nome_original' => $nomeOriginal,
                        'tamanho' => $arquivo->getSize(),
                    ];
                }
            }

            // Manter anexos existentes + adicionar novos
            $todosAnexos = array_merge($anexosExistentes, $novosAnexos);

            // Atualizar relatório
            $relatorio->update([
                'descricao_tecnica' => $request->descricao_tecnica,
                'descricao_financeira' => $request->descricao_financeira,
                'periodo_inicio' => $request->periodo_inicio,
                'periodo_fim' => $request->periodo_fim,
                'percentual_fisico' => $request->percentual_fisico,
                'valor_executado' => $request->valor_executado,
                'anexos' => $todosAnexos,
                'observacoes' => $request->observacoes,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Relatório atualizado com sucesso!',
                'redirect' => route('projetos.relatorios.show', [$projetoId, $relatorioId]),
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro ao atualizar relatório: '.$e->getMessage(),
            ], 500);
        }
    }

    /**
     * Excluir relatório
     */
    public function destroy($projetoId, $relatorioId)
    {
        $projeto = Projecto::findOrFail($projetoId);
        $relatorio = RelatorioExecucao::where('projeto_id', $projetoId)->findOrFail($relatorioId);

        // Verificar permissões (só o técnico pode excluir e só se estiver pendente)
        if ($projeto->tecnico_projectoFK !== Auth::id() || $relatorio->status !== 'pendente') {
            return response()->json(['success' => false, 'message' => 'Acesso negado'], 403);
        }

        try {
            // Excluir anexos do storage
            if ($relatorio->anexos) {
                foreach ($relatorio->anexos as $anexo) {
                    if (Storage::disk('public')->exists($anexo['path'])) {
                        Storage::disk('public')->delete($anexo['path']);
                    }
                }
            }

            $relatorio->delete();

            return response()->json([
                'success' => true,
                'message' => 'Relatório excluído com sucesso!',
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro ao excluir relatório: '.$e->getMessage(),
            ], 500);
        }
    }

    /**
     * Download de anexo
     */
    public function downloadAnexo($projetoId, $relatorioId, $anexoIndex)
    {
        $projeto = Projecto::findOrFail($projetoId);
        $relatorio = RelatorioExecucao::where('projeto_id', $projetoId)->findOrFail($relatorioId);

        // Verificar permissões
        if ($projeto->tecnico_projectoFK !== Auth::id() && $projeto->gestor_projectoFK !== Auth::id()) {
            abort(403, 'Acesso negado');
        }

        if (! isset($relatorio->anexos[$anexoIndex])) {
            abort(404, 'Anexo não encontrado');
        }

        $anexo = $relatorio->anexos[$anexoIndex];
        $pathCompleto = storage_path('app/public/'.$anexo['path']);

        if (! file_exists($pathCompleto)) {
            abort(404, 'Arquivo não encontrado');
        }

        return response()->download($pathCompleto, $anexo['nome_original']);
    }
}
