<?php

namespace App\Http\Controllers;

use App\Http\Requests\ProjetoRequest;
use App\Models\CronogramaFisico;
use App\Models\ProgramacaoFinanceira;
use App\Models\Projecto;
use App\Models\TipoProjecto;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Foundation\Auth\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class ProjectoController extends Controller
{
    /**
     * Exibir formulário de criação de projeto
     */
    public function index()
    {
        $tipoprojecto = TipoProjecto::all();
        $projecto = Projecto::latest()->first()->id ?? 0;
       
        
        

        return view('Clientes.CriarProjecto', ['projectos' => $tipoprojecto, 'codRef' => $projecto]);
    }

    /**
     * Listar projetos do usuário atual
     */
    public function show()
    {
        $projetos = Projecto::with(['tipoProjecto', 'cronogramasFisicos', 'programacoesFinanceiras', 'pagamentos'])
            ->where('tecnico_projectoFK', Auth::id())
            ->orderBy('created_at', 'desc')
            ->get();

        $projetosall = Projecto::with(['tipoProjecto', 'cronogramasFisicos', 'programacoesFinanceiras', 'pagamentos'])
            ->where('tecnico_projectoFK', Auth::id())
            ->orderBy('created_at', 'desc')
            ->paginate(5);

        return view('Clientes.ListarProjectos', compact('projetos', 'projetosall'));
    }

    /**
     * Exibir detalhes de um projeto específico
     */
    public function showDetails($id)
    {
        $projeto = Projecto::with([
            'tipoProjecto',
            'cronogramasFisicos',
            'programacoesFinanceiras',
            'relatoriosExecucao.usuario',
            'relatoriosExecucao.aprovador',
            'pagamentos.user'
        ])
            ->byUser(Auth::id())
            ->where('tecnico_projectoFK', Auth::id())
            ->findOrFail($id);

        return view('Clientes.DetalhesProjecto', compact('projeto'));
    }

    /**
     * Gerar resumo antes de salvar
     */
    public function getResumo(Request $request)
    {
        $dados = $request->all();

        // Calcular totais
        $totalFinanceiro = collect($dados['despesaValor'] ?? [])->sum();
        $totalAtividades = count($dados['actividadeDescricao'] ?? []);
        $totalProgramacoes = count($dados['despesaDescricao'] ?? []);

        return response()->json([
            'dados' => $dados,
            'totalFinanceiro' => number_format($totalFinanceiro, 2, ',', '.'),
            'totalAtividades' => $totalAtividades,
            'totalProgramacoes' => $totalProgramacoes,
        ]);
    }

    /**
     * Armazenar novo projeto
     */
    public function store(ProjetoRequest $request)
    {
        DB::beginTransaction();

        try {
            $request->validated();

            $idPromotora = User::where('id', Auth::id())->first()->id_entidade;

            // Upload de arquivos
            $facturaPath = null;
            $impactoPath = null;

            if ($request->hasFile('facturaP')) {
                $facturaPath = $request->file('facturaP')->store('facturas', 'public');
            }

            if ($request->hasFile('impactoSocial')) {
                $impactoPath = $request->file('impactoSocial')->store('impactos', 'public');
            }

            // Criar projeto
            $projecto = new Projecto;
            $projecto->codProjecto = $request->codProjecto;
            $projecto->nomeProjecto = $request->nomeProjecto;
            $projecto->tipoProjectFK = $request->tipoProjectFK;
            $projecto->descricaoObjectivo = $request->descObjectivo;
            $projecto->limiteorcamentalP = $request->limiteorcamentalP;
            $projecto->beneficiarioDP = $request->beneficiarioDP;
            $projecto->duracaoP = $request->duracaoP;
            $projecto->duracaoF = $request->duracaoF;
            $projecto->justificacao = $request->justificacao;
            $projecto->financeador = $request->ffinanciamento;
            $projecto->status = '0';
            $projecto->ambito = $request->ambito;
            $projecto->mFuncionamento = $request->mFuncionamento;
            $projecto->localizacao_projecto = $request->localizacao_projecto;
            $projecto->impactoSocial = $impactoPath;
            $projecto->empregosDirectos = $request->empregosDirectos;
            $projecto->kitsDistribuidos = $request->kitsDistribuidos;
            $projecto->aprovadoPor = $request->aprovadopor;
            $projecto->facturaPath = $facturaPath;
            $projecto->gestor_projectoFK = Auth::id();
            $projecto->tecnico_projectoFK = $request->tecnico ?? Auth::id();

            // Informações Bancarias
            $projecto->nomeInst = $request->nomeInst;
            $projecto->enderecoAgencia = $request->enderecoAgencia;
            $projecto->contactoAgencia = $request->contactoAgencia;
            $projecto->numeroConta = $request->numeroConta;
            $projecto->iban = $request->iban;
            $projecto->swif = $request->swif;
            $projecto->assinante1 = $request->Assinante1;
            $projecto->assinante2 = $request->Assinante2;
            $projecto->assinante3 = $request->Assinante3;
            $projecto->assinante4 = $request->Assinante4;
            $projecto->nif = $request->nif;

            $projecto->save();

            // Salvar cronograma físico

            if ($request->has('actividadeDescricao')) {
                foreach ($request->actividadeDescricao as $index => $descricao) {
                    CronogramaFisico::create([
                        'projeto_id' => $projecto->id,
                        'descricao' => $descricao,
                        'ano' => $request->actividadeAno[$index],
                        'data_inicio' => $request->actividadeDataInicio[$index],
                        'data_fim' => $request->actividadeDataFim[$index],
                        'percentual' => $request->actividadePercentual[$index],

                    ]);

                    // dd(CronogramaFisico::all());
                }
            }

            // Salvar programação financeira
            if ($request->has('despesaDescricao')) {
                foreach ($request->despesaDescricao as $index => $descricao) {
                    ProgramacaoFinanceira::create([
                        'projeto_id' => $projecto->id,
                        'descricao' => $descricao,
                        'ano' => $request->despesaAno[$index],
                        'valor' => $request->despesaValor[$index],
                        'data_inicial' => $request->despesaDatainicial[$index],
                        'data_final' => $request->despesaDatafinal[$index],

                    ]);

                }
            }

            DB::commit();

            return redirect()->route('Listar-Projecto')->with('msg', 'Projecto criado com sucesso!');

        } catch (\Exception $e) {
            DB::rollback();

            return back()->with('error', 'Erro ao salvar projecto: '.$e->getMessage());
        }
    }

    /**
     * Exibir formulário de edição
     */
    public function edit($id)
    {
        $projeto = Projecto::with(['cronogramasFisicos', 'programacoesFinanceiras'])
            ->byUser(Auth::id())
            ->findOrFail($id);

        $tipoprojecto = TipoProjecto::all();

        return view('Clientes.EditarProjecto', compact('projeto', 'tipoprojecto'));
    }

    /**
     * Atualizar projeto
     */
    public function update(Request $request, $id)
    {
        DB::beginTransaction();

        try {
            $projeto = Projecto::byUser(Auth::id())->findOrFail($id);

            // Atualizar dados do projeto
            $projeto->update($request->only([
                'nomeProjecto', 'tipoProjectFK', 'descricaoObjectivo',
                'limiteorcamentalP', 'beneficiarioDP', 'duracaoP', 'duracaoF',
                'justificacao', 'ambito', 'mFuncionamento', 'localizacao_projecto',
                'empregosDirectos', 'kitsDistribuidos',
            ]));

            // Atualizar cronograma físico
            if ($request->has('actividadeDescricao')) {
                $projeto->cronogramasFisicos()->delete();

                foreach ($request->actividadeDescricao as $index => $descricao) {
                    CronogramaFisico::create([
                        'projeto_id' => $projeto->id,
                        'descricao' => $descricao,
                        'ano' => $request->actividadeAno[$index],
                        'data_inicio' => $request->actividadeDataInicio[$index],
                        'data_fim' => $request->actividadeDataFim[$index],
                        'percentual' => $request->actividadePercentual[$index],

                    ]);
                }
            }

            // Atualizar programação financeira
            if ($request->has('despesaDescricao')) {
                $projeto->programacoesFinanceiras()->delete();

                foreach ($request->despesaDescricao as $index => $descricao) {
                    ProgramacaoFinanceira::create([
                        'projeto_id' => $projeto->id,
                        'descricao' => $descricao,
                        'ano' => $request->despesaAno[$index],
                        'valor' => $request->despesaValor[$index],
                        'data_prevista' => $request->despesaDataPrevista[$index],

                    ]);
                }
            }

            DB::commit();

            return redirect()->route('projetos.show', $projeto->id)
                ->with('msg', 'Projeto atualizado com sucesso!');

        } catch (\Exception $e) {
            DB::rollback();

            return back()->withErrors(['error' => 'Erro ao atualizar projeto: '.$e->getMessage()]);
        }
    }

    /**
     * Visualizar PDF do projeto (preview inline)
     */
    public function visualizarPDF($id)
    {
        try {
            $projeto = Projecto::with(['tipoProjecto', 'cronogramasFisicos', 'programacoesFinanceiras'])
                ->byUser(Auth::id())
                ->findOrFail($id);

            $data = [
                'projeto' => $projeto,
                'usuario' => Auth::user(),
                'dataExportacao' => now()->format('d/m/Y H:i:s'),
            ];

            $pdf = Pdf::loadView('pdf.ficha-projeto-final-simples', $data);
            $pdf->setPaper('A4', 'portrait');

            // Limpar caracteres inválidos do nome do arquivo
            $codigoLimpo = preg_replace('/[^a-zA-Z0-9_-]/', '_', $projeto->codProjecto);

            // Retornar PDF para visualização inline
            return $pdf->stream('Preview_'.$codigoLimpo.'.pdf');

        } catch (\Exception $e) {
            return back()->withErrors(['error' => 'Erro ao gerar PDF: '.$e->getMessage()]);
        }
    }

    /**
     * Exportar projeto para PDF (download)
     */
    public function exportarPDF($id)
    {
        try {
            $projeto = Projecto::with(['tipoProjecto', 'cronogramasFisicos', 'programacoesFinanceiras'])
                ->byUser(Auth::id())
                ->findOrFail($id);

            $data = [
                'projeto' => $projeto,
                'usuario' => Auth::user(),
                'dataExportacao' => now()->format('d/m/Y H:i:s'),
            ];

            $pdf = Pdf::loadView('pdf.ficha-projeto-final-simples', $data);
            $pdf->setPaper('A4', 'portrait');

            // Limpar caracteres inválidos do nome do arquivo
            $codigoLimpo = preg_replace('/[^a-zA-Z0-9_-]/', '_', $projeto->codProjecto);
            $nomeArquivo = 'Ficha_SIGEPE_'.$codigoLimpo.'_'.date('Y-m-d').'.pdf';

            return $pdf->download($nomeArquivo);

        } catch (\Exception $e) {
            return back()->withErrors(['error' => 'Erro ao gerar PDF: '.$e->getMessage()]);
        }
    }

    /**
     * Deletar projeto
     */
    public function destroy($id)
    {
        DB::beginTransaction();

        try {
            $projeto = Projecto::byUser(Auth::id())->findOrFail($id);

            // Excluir arquivos
            if ($projeto->facturaPath) {
                Storage::disk('public')->delete($projeto->facturaPath);
            }
            if ($projeto->impactoSocial) {
                Storage::disk('public')->delete($projeto->impactoSocial);
            }

            // Excluir relacionamentos
            $projeto->cronogramasFisicos()->delete();
            $projeto->programacoesFinanceiras()->delete();

            // Excluir projeto
            $projeto->delete();

            DB::commit();

            return redirect()->route('Listar-Projecto')
                ->with('msg', 'Projeto excluído com sucesso!');

        } catch (\Exception $e) {
            DB::rollback();

            return back()->withErrors(['error' => 'Erro ao excluir projeto: '.$e->getMessage()]);
        }
    }

    // ============== MÉTODOS ADMINISTRATIVOS ==============

    /**
     * Listar todos os projetos para administração
     */
    public function indexAdmin(Request $request)
    {
        $adminId = Auth::id();
        $query = Projecto::with(['tipoProjecto', 'adminResponsavel', 'tecnicoProjecto']);
        /*->where(function ($q) {
            // Controle de acesso: Status 0 (todos veem) OU admin responsável
            $q->where('status', '0');
             ->orWhere('admin_responsavel_id', $adminId);
        });*/

        // Filtros aplicados depois do controle de acesso
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('search')) {
            $query->where(function ($q) use ($request) {
                $q->where('nomeProjecto', 'like', '%'.$request->search.'%')
                    ->orWhere('codProjecto', 'like', '%'.$request->search.'%')
                    ->orWhere('descricaoObjectivo', 'like', '%'.$request->search.'%');
            });
        }

        if ($request->filled('tipo_projeto')) {
            $query->where('tipoProjectFK', $request->tipo_projeto);
        }

        if ($request->filled('data_inicio') && $request->filled('data_fim')) {
            $query->whereBetween('created_at', [$request->data_inicio, $request->data_fim]);
        }

        $projetos = $query->orderBy('created_at', 'desc')->paginate(12);
        $tiposProjecto = \App\Models\TipoProjecto::all();

        return view('Admin.GestaoProjectos', compact('projetos', 'tiposProjecto'));
    }

    /**
     * Visualizar detalhes do projeto para administrador
     */
    public function showAdmin($id)
    {
        $adminId = Auth::id();
        $projeto = Projecto::with([
            'tipoProjecto',
            'cronogramasFisicos',
            'programacoesFinanceiras',
            'adminResponsavel',
            'tecnicoProjecto',
        ])
            /*->where(function ($q) use ($adminId) {
                // Controle de acesso: Status 0 (todos veem) OU admin responsável
                $q->where('status', '0')
                    ->orWhere('admin_responsavel_id', $adminId);
            })*/
            ->findOrFail($id);

        return view('Admin.DetalhesProjectoAdmin', compact('projeto'));
    }

    /**
     * Atualizar status do projeto
     */
    public function atualizarStatus(Request $request, $id)
    {
        $request->validate([
            'novo_status' => 'required|integer|in:1,2,3,4,5',
            'observacoes' => 'nullable|string|max:1000',
        ]);

        DB::beginTransaction();

        try {
            $adminId = Auth::id();
            $projeto = Projecto::/*where(function ($q) use ($adminId) {
                // Controle de acesso: Status 0 (todos veem) OU admin responsável
                $q->where('status', '0')
                    ->orWhere('admin_responsavel_id', $adminId);
            })*/ findOrFail($id);

            // Verificar se o admin pode alterar o status
            if (! $projeto->podeAlterarStatus($adminId)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Você não tem permissão para alterar o status deste projeto.',
                ], 403);
            }

            // Verificar se a transição é válida
            $proximosStatus = $projeto->proximosStatusValidos();
            if (! in_array($request->novo_status, $proximosStatus)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Transição de status inválida.',
                ], 400);
            }

            // Atualizar projeto
            $projeto->update([
                'status' => $request->novo_status,
                'admin_responsavel_id' => $adminId,
                'data_mudanca_status' => now(),
            ]);

            // Log da ação (opcional - pode ser implementado depois)
            // LogAtividade::create([
            //     'usuario_id' => $adminId,
            //     'acao' => 'Status alterado',
            //     'descricao' => "Status do projeto {$projeto->codProjecto} alterado para {$projeto->status_formatado}",
            //     'projeto_id' => $id
            // ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Status atualizado com sucesso!',
                'novo_status' => $projeto->status_formatado,
                'cor_status' => $projeto->status_cor,
            ]);

        } catch (\Exception $e) {
            DB::rollback();

            return response()->json([
                'success' => false,
                'message' => 'Erro ao atualizar status: '.$e->getMessage(),
            ], 500);
        }
    }

    /**
     * Obter detalhes rápidos do projeto via AJAX
     */
    public function obterDetalhes($id)
    {
        try {
            $adminId = Auth::id();
            $projeto = Projecto::with(['tipoProjecto', 'adminResponsavel', 'tecnicoProjecto'])
                ->findOrFail($id);

            return response()->json([
                'success' => true,
                'projeto' => [
                    'id' => $projeto->id,
                    'codProjecto' => $projeto->codProjecto,
                    'nomeProjecto' => $projeto->nomeProjecto,
                    'descricaoObjectivo' => $projeto->descricaoObjectivo,
                    'limiteorcamentalP' => $projeto->limite_orcamental_formatado,
                    'tipoProjecto' => $projeto->tipoProjecto->descricao ?? 'N/A',
                    'tecnicoProjecto' => $projeto->tecnicoProjecto->name ?? 'N/A',
                    'adminResponsavel' => $projeto->adminResponsavel->name ?? 'N/A',
                    'status' => $projeto->status,
                    'created_at' => $projeto->created_at->format('d/m/Y H:i'),
                ],
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro ao carregar detalhes: '.$e->getMessage(),
            ], 500);
        }
    }

    /**
     * Exportar projeto para Excel
     */
    public function exportarExcel($id)
    {
        try {
            $adminId = Auth::id();
            $projeto = Projecto::with([
                'tipoProjecto',
                'cronogramasFisicos',
                'programacoesFinanceiras',
                'adminResponsavel',
                'tecnicoProjecto',
            ])
                ->where(function ($q) use ($adminId) {
                    // Controle de acesso: Status 0 (todos veem) OU admin responsável
                    $q->where('status', '0')
                        ->orWhere('admin_responsavel_id', $adminId);
                })
                ->findOrFail($id);

            // Dados do projeto
            $dadosProjeto = [
                ['Campo', 'Valor'],
                ['Código do Projeto', $projeto->codProjecto],
                ['Nome do Projeto', $projeto->nomeProjecto],
                ['Descrição/Objetivo', $projeto->descricaoObjectivo],
                ['Limite Orçamental', 'Kz '.number_format($projeto->limiteorcamentalP, 2, ',', '.')],
                ['Tipo de Projeto', $projeto->tipoProjecto->descricao ?? 'N/A'],
                ['Beneficiário', $projeto->beneficiarioDP],
                ['Duração do Projeto', $projeto->duracaoP.' meses'],
                ['Duração do Financiamento', $projeto->duracaoF.' meses'],
                ['Localização', $projeto->localizacao_projecto],
                ['Justificação', $projeto->justificacao],
                ['Aprovado Por', $projeto->aprovadoPor],
                ['Financiador', $projeto->financeador],
                ['Status', $projeto->status_formatado],
                ['Admin Responsável', $projeto->adminResponsavel->name ?? 'N/A'],
                ['Técnico Responsável', $projeto->tecnicoProjecto->name ?? 'N/A'],
                ['Data de Criação', $projeto->created_at->format('d/m/Y H:i')],
                ['Data de Atualização', $projeto->updated_at->format('d/m/Y H:i')],
            ];

            // Anexos disponíveis
            $anexos = [
                ['', ''], // Linha vazia
                ['ANEXOS DISPONÍVEIS', ''],
                ['Tipo de Anexo', 'Status'],
            ];

            $anexosDisponiveis = [
                'Url_contratoP' => 'Contrato do Projeto',
                'ambito' => 'Âmbito do Projeto',
                'impactoSocial' => 'Estudo de Impacto Social',
                'tecnicoEngenharia' => 'Parecer Técnico de Engenharia',
                'analiseEfectividade' => 'Análise de Efetividade',
                'viabilidadeEconomica' => 'Estudo de Viabilidade Econômica',
                'impactoAmbiental' => 'Estudo de Impacto Ambiental',
                'mitigacaoImpactos' => 'Plano de Mitigação de Impactos',
            ];

            foreach ($anexosDisponiveis as $campo => $titulo) {
                $status = ! empty($projeto->$campo) ? 'Disponível' : 'Não disponível';
                $anexos[] = [$titulo, $status];
            }

            // Cronograma físico
            $cronograma = [
                ['', ''], // Linha vazia
                ['CRONOGRAMA FÍSICO', ''],
                ['Ano', 'Descrição', 'Percentual', 'Período'],
            ];

            foreach ($projeto->cronogramasFisicos as $item) {
                $cronograma[] = [
                    $item->ano,
                    $item->descricao,
                    $item->percentual.'%',
                    $item->data_inicio.' à '.$item->data_fim,
                ];
            }

            // Programação financeira
            $programacao = [
                ['', ''], // Linha vazia
                ['PROGRAMAÇÃO FINANCEIRA', ''],
                ['Ano', 'Tipo de Despesa', 'Valor', 'Percentual'],
            ];

            $totalProgramacao = $projeto->programacoesFinanceiras->sum('valor');
            foreach ($projeto->programacoesFinanceiras as $item) {
                $percentual = $totalProgramacao > 0 ? round(($item->valor / $totalProgramacao) * 100, 2) : 0;
                $programacao[] = [
                    $item->ano,
                    $item->tipo_despesa,
                    'Kz '.number_format($item->valor, 2, ',', '.'),
                    $percentual.'%',
                ];
            }

            // Criar conteúdo CSV
            $csvContent = "\xEF\xBB\xBF"; // BOM para UTF-8

            // Adicionar dados do projeto
            foreach ($dadosProjeto as $linha) {
                $csvContent .= implode(';', $linha)."\n";
            }

            // Adicionar anexos
            foreach ($anexos as $linha) {
                $csvContent .= implode(';', $linha)."\n";
            }

            // Adicionar cronograma
            foreach ($cronograma as $linha) {
                $csvContent .= implode(';', $linha)."\n";
            }

            // Adicionar programação
            foreach ($programacao as $linha) {
                $csvContent .= implode(';', $linha)."\n";
            }

            // Nome do arquivo
            $nomeArquivo = 'Projeto_'.$projeto->codProjecto.'_'.date('Y-m-d_H-i-s').'.csv';

            // Retornar download
            return response($csvContent, 200, [
                'Content-Type' => 'text/csv; charset=UTF-8',
                'Content-Disposition' => 'attachment; filename="'.$nomeArquivo.'"',
                'Cache-Control' => 'no-cache, must-revalidate',
            ]);

        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Erro ao exportar projeto: '.$e->getMessage());
        }
    }

    /**
     * Exportar projeto para PDF - Área Administrativa
     */
    public function exportarPDFAdmin($id)
    {
        try {
            // Buscar o projeto com relacionamentos
            $projeto = Projecto::with([
                'cronogramasFisicos',
                'programacaoFinanceira',
                'tipoProjecto',
                'tecnicoProjecto',
                'adminResponsavel',
            ])->findOrFail($id);

            // Verificar controle de acesso administrativo
            if ($projeto->status != 0 && $projeto->admin_responsavel_id != Auth::id()) {
                return redirect()->route('admin.projetos.index')
                    ->with('error', 'Você não tem permissão para acessar este projeto.');
            }

            // Gerar PDF
            $pdf = Pdf::loadView('pdf.ficha-projeto-final-simples', compact('projeto'));
            $pdf->setPaper('A4', 'portrait');

            $nomeArquivo = 'Projeto_'.$projeto->id.'_'.date('Y-m-d_H-i-s').'.pdf';

            return $pdf->download($nomeArquivo);

        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Erro ao exportar PDF: '.$e->getMessage());
        }
    }

    /**
     * Visualizar PDF do projeto - Área Administrativa
     */
    public function visualizarPDFAdmin($id)
    {
        try {
            // Buscar o projeto com relacionamentos
            $projeto = Projecto::with([
                'cronogramasFisicos',
                'programacaoFinanceira',
                'tipoProjecto',
                'tecnicoProjecto',
                'adminResponsavel',
            ])->findOrFail($id);

            // Verificar controle de acesso administrativo
            if ($projeto->status != 0 && $projeto->admin_responsavel_id != Auth::id()) {
                return redirect()->route('admin.projetos.index')
                    ->with('error', 'Você não tem permissão para acessar este projeto.');
            }

            // Gerar PDF para visualização
            $pdf = Pdf::loadView('pdf.ficha-projeto-final-simples', compact('projeto'));
            $pdf->setPaper('A4', 'portrait');

            return $pdf->stream('Projeto_'.$projeto->id.'.pdf');

        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Erro ao visualizar PDF: '.$e->getMessage());
        }
    }

    /**
     * Registrar pagamento para projeto aprovado
     */
    public function registrarPagamento(Request $request, Projecto $projecto)
    {
        $request->validate([
            'descricao' => 'required|string|max:255',
            'valor' => 'required|numeric|min:0',
            'data' => 'required|date',
            'anexo' => 'nullable|file|mimes:pdf,jpg,png|max:2048'
        ]);

        $dados = $request->all();
        $dados['user_id'] = auth()->id();

        if ($request->hasFile('anexo')) {
            $dados['anexo'] = $request->file('anexo')->store('pagamentos');
        }

        $projecto->pagamentos()->create($dados);

        return back()->with('success', 'Pagamento registrado com sucesso!');
    }

    /**
     * Listar pagamentos de um projeto
     */
    public function listarPagamentos(Projecto $projecto)
    {
        return view('Admin.pagamentos', [
            'projecto' => $projecto,
            'pagamentos' => $projecto->pagamentos()->latest()->get()
        ]);
    }

    /**
     * Exportar listagem geral de projetos para PDF
     */
    public function exportarPDFListagemGeral(Request $request)
    {
        $adminId = Auth::id();
        $query = Projecto::with(['tipoProjecto', 'adminResponsavel', 'tecnicoProjecto']);

        // Aplicar os mesmos filtros do indexAdmin
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('search')) {
            $query->where(function ($q) use ($request) {
                $q->where('nomeProjecto', 'like', '%'.$request->search.'%')
                    ->orWhere('codProjecto', 'like', '%'.$request->search.'%')
                    ->orWhere('descricaoObjectivo', 'like', '%'.$request->search.'%');
            });
        }

        if ($request->filled('tipo_projeto')) {
            $query->where('tipoProjectFK', $request->tipo_projeto);
        }

        if ($request->filled('data_inicio') && $request->filled('data_fim')) {
            $query->whereBetween('created_at', [$request->data_inicio, $request->data_fim]);
        }

        $projetos = $query->orderBy('created_at', 'desc')->get();
        
        // Agrupar projetos por instituição
        $projetosPorInstituicao = $projetos->groupBy(function ($projeto) {
            return $projeto->tecnicoProjecto ? $projeto->tecnicoProjecto->name : 'Sem Técnico Responsável';
        });

        // Calcular totais
        $totalProjetos = $projetos->count();
        $custoTotal = $projetos->sum('limiteorcamentalP');
        $statusCounts = [
            'recebidos' => $projetos->where('status', 0)->count(),
            'em_analise' => $projetos->where('status', 1)->count(),
            'aprovados' => $projetos->where('status', 2)->count(),
            'rejeitados' => $projetos->where('status', 3)->count(),
            'em_execucao' => $projetos->where('status', 4)->count(),
            'concluidos' => $projetos->where('status', 5)->count(),
        ];

        $pdf = Pdf::loadView('pdf.gestao-projetos-listagem', compact(
            'projetosPorInstituicao',
            'totalProjetos',
            'custoTotal',
            'statusCounts'
        ));

        $pdf->setPaper('A4', 'landscape');

        return $pdf->download('listagem-projetos-' . date('Y-m-d-H-i-s') . '.pdf');
    }

    /**
     * Visualizar PDF da listagem geral de projetos
     */
    public function visualizarPDFListagemGeral(Request $request)
    {
        $adminId = Auth::id();
        $query = Projecto::with(['tipoProjecto', 'adminResponsavel', 'tecnicoProjecto']);

        // Aplicar os mesmos filtros do indexAdmin
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('search')) {
            $query->where(function ($q) use ($request) {
                $q->where('nomeProjecto', 'like', '%'.$request->search.'%')
                    ->orWhere('codProjecto', 'like', '%'.$request->search.'%')
                    ->orWhere('descricaoObjectivo', 'like', '%'.$request->search.'%');
            });
        }

        if ($request->filled('tipo_projeto')) {
            $query->where('tipoProjectFK', $request->tipo_projeto);
        }

        if ($request->filled('data_inicio') && $request->filled('data_fim')) {
            $query->whereBetween('created_at', [$request->data_inicio, $request->data_fim]);
        }

        $projetos = $query->orderBy('created_at', 'desc')->get();
        
        // Agrupar projetos por instituição
        $projetosPorInstituicao = $projetos->groupBy(function ($projeto) {
            return $projeto->tecnicoProjecto ? $projeto->tecnicoProjecto->name : 'Sem Técnico Responsável';
        });

        // Calcular totais
        $totalProjetos = $projetos->count();
        $custoTotal = $projetos->sum('limiteorcamentalP');
        $statusCounts = [
            'recebidos' => $projetos->where('status', 0)->count(),
            'em_analise' => $projetos->where('status', 1)->count(),
            'aprovados' => $projetos->where('status', 2)->count(),
            'rejeitados' => $projetos->where('status', 3)->count(),
            'em_execucao' => $projetos->where('status', 4)->count(),
            'concluidos' => $projetos->where('status', 5)->count(),
        ];

        $pdf = Pdf::loadView('pdf.gestao-projetos-listagem', compact(
            'projetosPorInstituicao',
            'totalProjetos',
            'custoTotal',
            'statusCounts'
        ));

        $pdf->setPaper('A4', 'landscape');

        return $pdf->stream('listagem-projetos.pdf');
    }

    /**
     * Exportar listagem geral de projetos para Excel
     */
    public function exportarExcelListagemGeral(Request $request)
    {
        $adminId = Auth::id();
        $query = Projecto::with(['tipoProjecto', 'adminResponsavel', 'tecnicoProjecto']);

        // Aplicar os mesmos filtros do indexAdmin
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('search')) {
            $query->where(function ($q) use ($request) {
                $q->where('nomeProjecto', 'like', '%'.$request->search.'%')
                    ->orWhere('codProjecto', 'like', '%'.$request->search.'%')
                    ->orWhere('descricaoObjectivo', 'like', '%'.$request->search.'%');
            });
        }

        if ($request->filled('tipo_projeto')) {
            $query->where('tipoProjectFK', $request->tipo_projeto);
        }

        if ($request->filled('data_inicio') && $request->filled('data_fim')) {
            $query->whereBetween('created_at', [$request->data_inicio, $request->data_fim]);
        }

        $projetos = $query->orderBy('created_at', 'desc')->get();

        // Preparar dados para CSV
        $filename = 'listagem-projetos-' . date('Y-m-d-H-i-s') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv; charset=UTF-8',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];

        $callback = function() use ($projetos) {
            $file = fopen('php://output', 'w');
            
            // Adicionar BOM para UTF-8
            fprintf($file, chr(0xEF).chr(0xBB).chr(0xBF));
            
            // Cabeçalhos
            fputcsv($file, [
                'Código',
                'Nome do Projeto',
                'Tipo',
                'Instituição/Técnico',
                'Status',
                'Custo Total (Kz)',
                'Data de Submissão',
                'Descrição/Objetivo'
            ], ';');

            // Dados dos projetos
            foreach ($projetos as $projeto) {
                $status = '';
                switch ($projeto->status) {
                    case 0: $status = 'Recebido'; break;
                    case 1: $status = 'Em Análise'; break;
                    case 2: $status = 'Aprovado'; break;
                    case 3: $status = 'Rejeitado'; break;
                    case 4: $status = 'Em Execução'; break;
                    case 5: $status = 'Concluído'; break;
                    default: $status = 'N/A'; break;
                }

                fputcsv($file, [
                    $projeto->codProjecto ?? 'N/A',
                    $projeto->nomeProjecto ?? 'N/A',
                    $projeto->tipoProjecto->name ?? 'N/A',
                    $projeto->tecnicoProjecto->name ?? 'Sem Técnico',
                    $status,
                    number_format($projeto->limiteorcamentalP ?? 0, 2, ',', '.'),
                    $projeto->created_at ? $projeto->created_at->format('d/m/Y') : 'N/A',
                    $projeto->descricaoObjectivo ?? 'N/A'
                ], ';');
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}
