<?php

namespace App\Http\Controllers;

use App\Models\ProcessamentoSalario;
use App\Models\SalarioProcessado;
use App\Models\Funcionario;
use App\Models\Departamento;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use App\Mail\ReciboSalarioMail;
use Barryvdh\DomPDF\Facade\Pdf;

class ProcessamentoSalarioController extends Controller
{
    public function index(Request $request)
    {
        $query = ProcessamentoSalario::with('salariosProcessados');
        
        if ($request->ano) $query->where('ano', $request->ano);
        if ($request->mes) $query->where('mes', $request->mes);
        if ($request->status) $query->where('status', $request->status);

        $processamentos = $query->orderBy('dataProcessamento', 'desc')->paginate(10);
        $departamentos = Departamento::ativos()->get();
        
        $totalProcessado = ProcessamentoSalario::where('status', 'Concluído')
            ->whereYear('dataProcessamento', date('Y'))
            ->get()->sum(fn($p) => $p->totalSalarios());

        return view('Administracao.ProcessamentoSalario', compact('processamentos', 'departamentos', 'totalProcessado'));
    }

    public function create(Request $request)
    {
        $request->validate([
            'descricao' => 'required|max:100',
            'ano' => 'required|integer',
            'mes' => 'required|integer|min:1|max:12',
            'unidadePagadora' => 'required|max:200',
            'unidadeProcessadora' => 'required|max:200',
        ]);

        $funcionarios = Funcionario::with(['remuneracoes', 'funcao', 'grupoPessoal', 'departamento'])
            ->where('estado', 'Ativo')
            ->when($request->id_departamento, fn($q) => $q->where('id_departamento', $request->id_departamento))
            ->get();

        $dadosFuncionarios = [];
        foreach ($funcionarios as $func) {
            $sb = $st = $sa = $sf = $so = $ab = 0;
            $tc = $desc = '';

            foreach ($func->remuneracoes()->wherePivot('status', 'Ativo')->get() as $rem) {
                $v = $rem->pivot->valor;
                $t = strtolower($rem->tipo);
                
                if (str_contains($t, 'salário') || str_contains($t, 'base')) {
                    $sb += $v; $tc = $rem->tipo; $desc = $rem->descricao;
                } elseif (str_contains($t, 'transporte')) $st += $v;
                elseif (str_contains($t, 'alimenta')) $sa += $v;
                elseif (str_contains($t, 'férias') || str_contains($t, 'ferias')) $sf += $v;
                elseif (str_contains($t, 'abono') || str_contains($t, 'familiar')) $ab += $v;
                else $so += $v;
            }

            $bruto = $sb + $ab + $st + $sa + $sf + $so;
            $descontoCalculado = $bruto * 0.07;

            $dadosFuncionarios[] = [
                'id' => $func->id,
                'nome' => $func->nomeCompleto,
                'nbi' => $func->numeroDocumento,
                'funcao' => $func->funcao->nomeFuncao ?? 'N/A',
                'grupoPessoal' => $func->grupoPessoal->nomeGrupo ?? 'N/A',
                'departamento' => $func->departamento->nomeDepartamento ?? 'N/A',
                'salarioBase' => $sb,
                'abonoFamiliar' => $ab,
                'subTransporte' => $st,
                'subAlimentacao' => $sa,
                'subFerias' => $sf,
                'subOutros' => $so,
                'descontoCalculado' => $descontoCalculado,
                'totalBruto' => $bruto,
                'tipoClasse' => $tc ?: 'Geral',
                'descricao' => $desc ?: 'Salário'
            ];
        }

        $departamentos = Departamento::ativos()->get();
        $parametros = $request->all();

        return view('Administracao.RevisarProcessamentoSalario', compact('dadosFuncionarios', 'parametros', 'departamentos'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'descricao' => 'required|max:100',
            'ano' => 'required|integer',
            'mes' => 'required|integer|min:1|max:12',
            'unidadePagadora' => 'required|max:200',
            'unidadeProcessadora' => 'required|max:200',
            'funcionarios' => 'required|array',
        ]);

        DB::beginTransaction();
        try {
            $processamento = ProcessamentoSalario::create([
                'descricao' => $request->descricao,
                'dataProcessamento' => now(),
                'ano' => $request->ano,
                'mes' => $request->mes,
                'unidadePagadora' => $request->unidadePagadora,
                'unidadeProcessadora' => $request->unidadeProcessadora,
                'status' => 'Processando'
            ]);

            $emailsEnviados = 0;
            $emailsFalhados = 0;

            foreach ($request->funcionarios as $idx => $funcData) {
                $observacao = $request->input("observacoes.{$idx}", '');

                // Buscar e-mail do funcionário pelo ID
                $funcionario = Funcionario::find($funcData['id']);
                $emailFuncionario = $funcionario ? $funcionario->email : null;

                // Todos os valores já vêm editados do formulário
                $salarioBase = (float)($funcData['salarioBase'] ?? 0);
                $abonoFamiliar = (float)($funcData['abonoFamiliar'] ?? 0);
                $subTransporte = (float)($funcData['subTransporte'] ?? 0);
                $subAlimentacao = (float)($funcData['subAlimentacao'] ?? 0);
                $subFerias = (float)($funcData['subFerias'] ?? 0);
                $subOutros = (float)($funcData['subOutros'] ?? 0);
                $desconto = (float)($funcData['descontoCalculado'] ?? 0);
                $abonoExtra = (float)($funcData['acrescimoExtra'] ?? 0);

                // Calcular totais
                $bruto = $salarioBase + $abonoFamiliar + $subTransporte + 
                         $subAlimentacao + $subFerias + $subOutros + $abonoExtra;
                $total = $bruto - $desconto;

                // Criar registro de salário processado
                $salarioProcessado = SalarioProcessado::create([
                    'nomeTrabalhador' => $funcData['nome'],
                    'nbi' => $funcData['nbi'],
                    'email_funcionario' => $emailFuncionario,
                    'funcao' => $funcData['funcao'],
                    'grupoPessoal' => $funcData['grupoPessoal'],
                    'tipoClasse' => $funcData['tipoClasse'],
                    'descricao' => $observacao ?: $funcData['descricao'],
                    'salarioBase' => $salarioBase,
                    'abonoFamiliar' => $abonoFamiliar,
                    'subTransporte' => $subTransporte,
                    'subAlimentacao' => $subAlimentacao,
                    'subFerias' => $subFerias,
                    'subOutros' => $subOutros,
                    'desconto' => $desconto,
                    'acrescimoExtra' => $abonoExtra,
                    'total' => $total,
                    'status_envio_email' => $emailFuncionario ? 'Pendente' : 'Sem Email',
                    'processamento_salarios_id' => $processamento->id
                ]);

                // Tentar enviar e-mail se o funcionário tiver e-mail
                if ($emailFuncionario && filter_var($emailFuncionario, FILTER_VALIDATE_EMAIL)) {
                    try {
                        Mail::to($emailFuncionario)->send(new ReciboSalarioMail($salarioProcessado, $processamento));
                        
                        $salarioProcessado->update([
                            'status_envio_email' => 'Enviado',
                            'data_envio_email' => now(),
                            'erro_envio_email' => null
                        ]);
                        $emailsEnviados++;
                    } catch (\Exception $e) {
                        $salarioProcessado->update([
                            'status_envio_email' => 'Falhou',
                            'erro_envio_email' => $e->getMessage()
                        ]);
                        $emailsFalhados++;
                    }
                }
            }

            $processamento->update(['status' => 'Concluído']);
            DB::commit();

            $mensagem = 'Processamento concluído! ' . count($request->funcionarios) . ' funcionários processados.';
            if ($emailsEnviados > 0) {
                $mensagem .= ' | 📧 ' . $emailsEnviados . ' e-mail(s) enviado(s) com sucesso.';
            }
            if ($emailsFalhados > 0) {
                $mensagem .= ' | ⚠️ ' . $emailsFalhados . ' e-mail(s) falharam.';
            }

            return redirect()->route('processamento-salarios.index')
                ->with('success', $mensagem);
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Erro: ' . $e->getMessage())->withInput();
        }
    }

    private function processar($func, $pid)
    {
        $sb = $st = $sa = $sf = $so = $ab = 0;
        $tc = $desc = '';

        foreach ($func->remuneracoes()->wherePivot('status', 'Ativo')->get() as $rem) {
            $v = $rem->pivot->valor;
            $t = strtolower($rem->tipo);
            
            if (str_contains($t, 'salário') || str_contains($t, 'base')) {
                $sb += $v; $tc = $rem->tipo; $desc = $rem->descricao;
            } elseif (str_contains($t, 'transporte')) $st += $v;
            elseif (str_contains($t, 'alimenta')) $sa += $v;
            elseif (str_contains($t, 'férias') || str_contains($t, 'ferias')) $sf += $v;
            elseif (str_contains($t, 'abono') || str_contains($t, 'familiar')) $ab += $v;
            else $so += $v;
        }

        $bruto = $sb + $ab + $st + $sa + $sf + $so;
        $desc_val = $bruto * 0.07;

        SalarioProcessado::create([
            'nomeTrabalhador' => $func->nomeCompleto,
            'nbi' => $func->numeroDocumento,
            'funcao' => $func->funcao->nomeFuncao ?? 'N/A',
            'grupoPessoal' => $func->grupoPessoal->nomeGrupo ?? 'N/A',
            'tipoClasse' => $tc ?: 'Geral',
            'descricao' => $desc ?: 'Salário',
            'salarioBase' => $sb,
            'abonoFamiliar' => $ab,
            'subTransporte' => $st,
            'subAlimentacao' => $sa,
            'subFerias' => $sf,
            'subOutros' => $so,
            'desconto' => $desc_val,
            'acrescimoExtra' => 0,
            'total' => $bruto - $desc_val,
            'processamento_salarios_id' => $pid
        ]);
    }

    public function show($id)
    {
        $processamento = ProcessamentoSalario::with('salariosProcessados')->findOrFail($id);
        return view('Administracao.DetalheProcessamentoSalario', compact('processamento'));
    }

    /**
     * Reenviar e-mail com recibo para um funcionário específico
     */
    public function reenviarEmail($processamentoId, $salarioId)
    {
        try {
            $processamento = ProcessamentoSalario::findOrFail($processamentoId);
            $salario = SalarioProcessado::findOrFail($salarioId);

            // Verificar se tem e-mail
            if (!$salario->email_funcionario || !filter_var($salario->email_funcionario, FILTER_VALIDATE_EMAIL)) {
                return back()->with('error', 'Funcionário não possui e-mail válido cadastrado.');
            }

            // Atualizar status para pendente
            $salario->update(['status_envio_email' => 'Pendente']);

            // Tentar enviar e-mail
            try {
                Mail::to($salario->email_funcionario)->send(new ReciboSalarioMail($salario, $processamento));
                
                $salario->update([
                    'status_envio_email' => 'Enviado',
                    'data_envio_email' => now(),
                    'erro_envio_email' => null
                ]);

                return back()->with('success', '📧 E-mail reenviado com sucesso para ' . $salario->nomeTrabalhador . '!');
            } catch (\Exception $e) {
                $salario->update([
                    'status_envio_email' => 'Falhou',
                    'erro_envio_email' => $e->getMessage()
                ]);

                return back()->with('error', '❌ Falha ao enviar e-mail: ' . $e->getMessage());
            }
        } catch (\Exception $e) {
            return back()->with('error', 'Erro: ' . $e->getMessage());
        }
    }

    /**
     * Sincronizar e-mails dos funcionários para registros sem e-mail
     */
    public function sincronizarEmails($processamentoId)
    {
        try {
            $processamento = ProcessamentoSalario::with('salariosProcessados')->findOrFail($processamentoId);
            $atualizados = 0;

            foreach ($processamento->salariosProcessados as $salario) {
                // Só atualiza se não tiver e-mail cadastrado
                if (!$salario->email_funcionario) {
                    // Buscar funcionário pelo NBI
                    $funcionario = Funcionario::where('numeroColaborador', $salario->nbi)
                        ->orWhere('numeroDocumento', $salario->nbi)
                        ->first();

                    if ($funcionario && $funcionario->email) {
                        $salario->update([
                            'email_funcionario' => $funcionario->email,
                            'status_envio_email' => 'Pendente'
                        ]);
                        $atualizados++;
                    }
                }
            }

            if ($atualizados > 0) {
                return back()->with('success', '✅ ' . $atualizados . ' e-mail(s) sincronizado(s) com sucesso!');
            } else {
                return back()->with('info', 'Nenhum e-mail para sincronizar.');
            }
        } catch (\Exception $e) {
            return back()->with('error', 'Erro ao sincronizar: ' . $e->getMessage());
        }
    }

    /**
     * Reenviar todos os e-mails pendentes ou que falharam
     */
    public function reenviarTodosEmails($processamentoId)
    {
        try {
            $processamento = ProcessamentoSalario::with('salariosProcessados')->findOrFail($processamentoId);
            
            // Buscar salários com status pendente, falhou ou sem email (mas que tenham e-mail agora)
            $salarios = $processamento->salariosProcessados->filter(function($sal) {
                return in_array($sal->status_envio_email, ['Pendente', 'Falhou', 'Sem Email']) 
                    && $sal->email_funcionario 
                    && filter_var($sal->email_funcionario, FILTER_VALIDATE_EMAIL);
            });

            if ($salarios->isEmpty()) {
                return back()->with('error', 'Não há e-mails pendentes para reenviar.');
            }

            $enviados = 0;
            $falhados = 0;

            foreach ($salarios as $salario) {
                $salario->update(['status_envio_email' => 'Pendente']);

                try {
                    Mail::to($salario->email_funcionario)->send(new ReciboSalarioMail($salario, $processamento));
                    
                    $salario->update([
                        'status_envio_email' => 'Enviado',
                        'data_envio_email' => now(),
                        'erro_envio_email' => null
                    ]);
                    $enviados++;
                } catch (\Exception $e) {
                    $salario->update([
                        'status_envio_email' => 'Falhou',
                        'erro_envio_email' => $e->getMessage()
                    ]);
                    $falhados++;
                }
            }

            $mensagem = '📧 Processo concluído! ';
            if ($enviados > 0) {
                $mensagem .= $enviados . ' e-mail(s) enviado(s) com sucesso. ';
            }
            if ($falhados > 0) {
                $mensagem .= '⚠️ ' . $falhados . ' e-mail(s) falharam.';
            }

            return back()->with('success', $mensagem);
        } catch (\Exception $e) {
            return back()->with('error', 'Erro: ' . $e->getMessage());
        }
    }

    public function exportarPDFGeral($id)
    {
        $processamento = ProcessamentoSalario::with('salariosProcessados')->findOrFail($id);
        $pdf = Pdf::loadView('pdf.folha-pagamento-geral', compact('processamento'))->setPaper('a4', 'landscape');
        
        // Remover caracteres inválidos do nome do arquivo
        $nomeArquivo = str_replace(['/', '\\'], '-', $processamento->periodoReferencia);
        return $pdf->download('folha-pagamento-' . $nomeArquivo . '.pdf');
    }

    public function exportarPDFIndividual($processamentoId, $salarioId)
    {
        $salario = SalarioProcessado::findOrFail($salarioId);
        $processamento = ProcessamentoSalario::findOrFail($processamentoId);
        $pdf = Pdf::loadView('pdf.recibo-salario', compact('salario', 'processamento'));
        return $pdf->download('recibo-' . $salario->nbi . '.pdf');
    }
}
