<?php

namespace App\Http\Controllers;

use App\Models\Projecto;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;

class HomeController extends Controller
{
    //
    public function index()
    {
        // Verificar se é admin (dashboard administrativo) ou promotor
        $user = Auth::user();

        if ($user->role === 'admin') {
            return $this->dashboardPromotor();
        } else {
            return $this->dashboardAdmin();
        }
    }

    private function dashboardPromotor()
    {
        // Obter o ID do usuário logado
        $userId = Auth::id();

        // Buscar todos os projetos do promotor
        $projectos = Projecto::get();

        // Contadores por status
        $totalProjectos = $projectos->count();
        $projectosPorStatus = [
            0 => $projectos->where('status', 0)->count(),
            1 => $projectos->where('status', 1)->count(),
            2 => $projectos->where('status', 2)->count(),
            3 => $projectos->where('status', 3)->count(),
            4 => $projectos->where('status', 4)->count(),
            5 => $projectos->where('status', 5)->count(),
        ];

        // Valores por status
        $valoresPorStatus = [
            0 => $projectos->where('status', 0)->sum('limiteorcamentalP'),
            1 => $projectos->where('status', 1)->sum('limiteorcamentalP'),
            2 => $projectos->where('status', 2)->sum('limiteorcamentalP'),
            3 => $projectos->where('status', 3)->sum('limiteorcamentalP'),
            4 => $projectos->where('status', 4)->sum('limiteorcamentalP'),
            5 => $projectos->where('status', 5)->sum('limiteorcamentalP'),
        ];

        // Projetos recentes
        $projectosRecentes = Projecto::all();

        // Taxas de progresso
        $taxaAprovacao = $totalProjectos > 0 ?
            round((($projectosPorStatus[5]) / $totalProjectos) * 100) : 0;

        // Para promotores, criar métricas vazias para evitar erros na view
        $metricas = [
            'totalRecebidosMes' => 0,
            'projetosAprovadosMes' => 0,
            'percentualAprovados' => 0,
            'projetosFinanciadosMes' => 0,
            'percentualFinanciados' => 0,
            'percentualDentroPrazo' => 0,
            'percentualForaPrazo' => 0,
            'projetosRejeitadosMes' => 0,
            'percentualRejeitados' => 0,
            'projetosOperacionais' => 0,
            'pareceresEmitidosMes' => 0,
            'pareceresDentroPrazo' => 0,
            'pareceresForaPrazo' => 0,
            'projetosRecebidosPorAdmin' => [],
        ];

        return view('Administracao.home', compact(
            'totalProjectos', 'projectosPorStatus', 'valoresPorStatus',
            'projectosRecentes', 'taxaAprovacao', 'metricas'
        ));
    }

    private function dashboardAdmin()
    {
        // Obter o ID do usuário logado
        $userId = Auth::id();

        // Buscar todos os projetos do promotor
        $projectos = Projecto::get();

        // Contadores por status
        $totalProjectos = $projectos->count();
        $projectosPorStatus = [
            0 => $projectos->where('status', 0)->count(),
            1 => $projectos->where('status', 1)->count(),
            2 => $projectos->where('status', 2)->count(),
            3 => $projectos->where('status', 3)->count(),
            4 => $projectos->where('status', 4)->count(),
            5 => $projectos->where('status', 5)->count(),
        ];

        // Valores por status
        $valoresPorStatus = [
            0 => $projectos->where('status', 0)->sum('limiteorcamentalP'),
            1 => $projectos->where('status', 1)->sum('limiteorcamentalP'),
            2 => $projectos->where('status', 2)->sum('limiteorcamentalP'),
            3 => $projectos->where('status', 3)->sum('limiteorcamentalP'),
            4 => $projectos->where('status', 4)->sum('limiteorcamentalP'),
            5 => $projectos->where('status', 5)->sum('limiteorcamentalP'),
        ];

        // Projetos recentes
        $projectosRecentes = Projecto::all();

        // Taxa de aprovação
        $taxaAprovacao = $totalProjectos > 0 ?
            round((($projectosPorStatus['2'] + $projectosPorStatus['4'] + $projectosPorStatus['5']) / $totalProjectos) * 100) : 0;

        // ========== MÉTRICAS E IMPACTOS ==========
        $dataAtual = Carbon::now();
        $inicioMes = $dataAtual->copy()->startOfMonth();
        $fimMes = $dataAtual->copy()->endOfMonth();
        $inicioAno = $dataAtual->copy()->startOfYear();

        // 1. Projetos recebidos pelos admin no período
        $projetosRecebidosPorAdmin = projecto::all();

        $totalRecebidosMes = Projecto::whereBetween('created_at', [$inicioMes, $fimMes])->count();

        // 2. Projectos aprovados no período (status 2)
        $projetosAprovadosMes = Projecto::where('status', '2')
            ->whereBetween('data_mudanca_status', [$inicioMes, $fimMes])
            ->count();
        $percentualAprovados = $totalRecebidosMes > 0 ?
            round(($projetosAprovadosMes / $totalRecebidosMes) * 100, 1) : 0;

        // 3. Projetos financiados (Em Execução + Concluídos)
        $projetosFinanciadosMes = Projecto::whereIn('status', ['4', '5'])
            ->whereBetween('data_mudanca_status', [$inicioMes, $fimMes])
            ->count();
        $percentualFinanciados = $totalRecebidosMes > 0 ?
            round(($projetosFinanciadosMes / $totalRecebidosMes) * 100, 1) : 0;

        // 4. Projetos financiados dentro/fora do prazo
        // Considerando 30 dias como prazo padrão para análise
        $projetosFinanciadosTotal = Projecto::whereIn('status', ['4', '5'])->get();
        $dentroRealizadosNoPrazo = 0;
        $foraRealizadosNoPrazo = 0;

        foreach ($projetosFinanciadosTotal as $projeto) {
            if ($projeto->created_at && $projeto->data_mudanca_status) {
                $diasProcessamento = Carbon::parse($projeto->created_at)
                    ->diffInDays(Carbon::parse($projeto->data_mudanca_status));
                if ($diasProcessamento <= 30) {
                    $dentroRealizadosNoPrazo++;
                } else {
                    $foraRealizadosNoPrazo++;
                }
            }
        }

        $totalFinanciados = $projetosFinanciadosTotal->count();
        $percentualDentroPrazo = $totalFinanciados > 0 ?
            round(($dentroRealizadosNoPrazo / $totalFinanciados) * 100, 1) : 0;
        $percentualForaPrazo = $totalFinanciados > 0 ?
            round(($foraRealizadosNoPrazo / $totalFinanciados) * 100, 1) : 0;

        // 5. Projetos rejeitados
        $projetosRejeitadosMes = Projecto::where('status', '3')
            ->whereBetween('data_mudanca_status', [$inicioMes, $fimMes])
            ->count();

        // dd($projetosRejeitadosMes.' -----'.$inicioMes.' ----- '.$fimMes);

        $percentualRejeitados = $totalRecebidosMes > 0 ?
            round(($projetosRejeitadosMes / $totalRecebidosMes) * 100, 1) : 0;

        // 6. Total de projetos financiados operacionais (Em Execução)
        $projetosOperacionais = Projecto::where('status', ['4', '5'])->count();

        // 7. Pareceres emitidos (mudanças de status que representam pareceres)
        $pareceresEmitidosMes = Projecto::whereIn('status', ['1', '2', '3', '4', '5'])
            ->whereBetween('data_mudanca_status', [$inicioMes, $fimMes])
            ->count();

        // Pareceres dentro/fora do prazo (considerando 15 dias para parecer)
        $pareceresDentroPrazo = 0;
        $pareceresForaPrazo = 0;

        $projetosComParecer = Projecto::whereIn('status', ['1', '2', '3', '4', '5'])
            ->whereNotNull('data_mudanca_status')
            ->get();

        foreach ($projetosComParecer as $projeto) {
            if ($projeto->created_at && $projeto->data_mudanca_status) {
                $diasParecer = Carbon::parse($projeto->created_at)
                    ->diffInDays(Carbon::parse($projeto->data_mudanca_status));
                if ($diasParecer <= 15) {
                    $pareceresDentroPrazo++;
                } else {
                    $pareceresForaPrazo++;
                }
            }
        }

        $totalProjectosCarteira = Projecto::where('admin_responsavel_id', Auth::id())->sum('limiteorcamentalP');

        session(['totalProjectosCarteira' => $totalProjectosCarteira]);

        // Organizar métricas
        $metricas = [

            'totalRecebidosMes' => $totalRecebidosMes,
            'projetosAprovadosMes' => $projetosAprovadosMes,
            'percentualAprovados' => $percentualAprovados,
            'projetosFinanciadosMes' => $projetosFinanciadosMes,
            'percentualFinanciados' => $percentualFinanciados,
            'percentualDentroPrazo' => $percentualDentroPrazo,
            'percentualForaPrazo' => $percentualForaPrazo,
            'projetosRejeitadosMes' => $projetosRejeitadosMes,
            'percentualRejeitados' => $percentualRejeitados,
            'projetosOperacionais' => $projetosOperacionais,
            'pareceresEmitidosMes' => $pareceresEmitidosMes,
            'pareceresDentroPrazo' => $pareceresDentroPrazo,
            'pareceresForaPrazo' => $pareceresForaPrazo,
            'projetosRecebidosPorAdmin' => $projetosRecebidosPorAdmin,
        ];

        // Organizar somatorios por status
        $somatorioPorStatus = [
            0 => $projectos->where('status', 0)->sum('limiteorcamentalP'),
            1 => $projectos->where('status', 1)->sum('limiteorcamentalP'),
            2 => $projectos->where('status', 2)->sum('limiteorcamentalP'),
            3 => $projectos->where('status', 3)->sum('limiteorcamentalP'),
            4 => $projectos->where('status', 4)->sum('limiteorcamentalP'),
            5 => $projectos->where('status', 5)->sum('limiteorcamentalP'),
            6 => $projectos->whereIn('status', [0, 1, 2, 3, 4, 5])->sum('limiteorcamentalP'),
        ];

        // Todas as variáveis são enviadas para a view
        return view('Administracao.home', compact(
            'totalProjectos', 'projectosPorStatus', 'valoresPorStatus', 'somatorioPorStatus',
            'projectosRecentes', 'taxaAprovacao', 'metricas'
        ));
    }
}
