<?php

namespace App\Http\Controllers;

use App\Models\GrupoPessoais;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class GrupoPessoaisController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        // Parâmetros de filtro
        $nomeGrupo = $request->input('nomeGrupo');
        $estado = $request->input('estado');

        // Query base
        $query = GrupoPessoais::query();

        // Aplicar filtros se fornecidos
        if ($nomeGrupo) {
            $query->where('nomeGrupo', 'like', '%' . $nomeGrupo . '%');
        }

        if ($estado !== null && $estado !== '') {
            $query->where('estado', $estado);
        }

        // Ordenar e paginar resultados
        $grupoPessoais = $query->orderBy('id', 'desc')
                      ->paginate(10)
                      ->withQueryString();

        return view('Administracao.GrupoPessoais', compact('grupoPessoais'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('Administracao.CriarGrupoPessoais');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // Validação
        $validator = Validator::make($request->all(), [
            'nomeGrupo' => 'required|string|max:255|unique:grupo_pessoais,nomeGrupo',
            'ObservacaoGrupo' => 'nullable|string|max:1000',
        ], [
            'nomeGrupo.required' => 'O nome do grupo pessoal é obrigatório',
            'nomeGrupo.max' => 'O nome do grupo pessoal não pode ter mais de 255 caracteres',
            'nomeGrupo.unique' => 'Este grupo pessoal já existe',
            'ObservacaoGrupo.max' => 'A observação não pode ter mais de 1000 caracteres'
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // Criar o grupo pessoal
        GrupoPessoais::create([
            'nomeGrupo' => $request->nomeGrupo,
            'ObservacaoGrupo' => $request->ObservacaoGrupo,
            'estado' => 'Ativo' // Ativo por padrão
        ]);

        return redirect()->route('grupo-pessoais.index')
            ->with('success', 'Grupo pessoal criado com sucesso!');
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        $grupoPessoal = GrupoPessoais::findOrFail($id);
        return view('Administracao.DetalheGrupoPessoais', compact('grupoPessoal'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        $grupoPessoal = GrupoPessoais::findOrFail($id);
        return view('Administracao.EditarGrupoPessoais', compact('grupoPessoal'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $grupoPessoal = GrupoPessoais::findOrFail($id);

        // Validação
        $validator = Validator::make($request->all(), [
            'nomeGrupo' => 'required|string|max:255|unique:grupo_pessoais,nomeGrupo,' . $id,
            'ObservacaoGrupo' => 'nullable|string|max:1000',
            'estado' => 'required|in:Ativo,Inativo',
        ], [
            'nomeGrupo.required' => 'O nome do grupo pessoal é obrigatório',
            'nomeGrupo.max' => 'O nome do grupo pessoal não pode ter mais de 255 caracteres',
            'nomeGrupo.unique' => 'Este grupo pessoal já existe',
            'ObservacaoGrupo.max' => 'A observação não pode ter mais de 1000 caracteres',
            'estado.required' => 'O estado é obrigatório',
            'estado.in' => 'Estado deve ser Ativo ou Inativo'
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // Atualizar o grupo pessoal
        $grupoPessoal->nomeGrupo = $request->nomeGrupo;
        $grupoPessoal->ObservacaoGrupo = $request->ObservacaoGrupo;
        $grupoPessoal->estado = $request->estado;
        $grupoPessoal->save();

        return redirect()->route('grupo-pessoais.index')
            ->with('success', 'Grupo pessoal atualizado com sucesso!');
    }

    /**
     * Toggle the status of the specified resource.
     */
    public function toggleStatus($id)
    {
        $grupoPessoal = GrupoPessoais::findOrFail($id);

        // Alternar estado (Ativo <-> Inativo)
        $grupoPessoal->estado = ($grupoPessoal->estado == 'Ativo') ? 'Inativo' : 'Ativo';
        $grupoPessoal->save();

        $statusText = $grupoPessoal->estado == 'Ativo' ? 'ativado' : 'desativado';

        return redirect()->route('grupo-pessoais.index')
            ->with('success', "Grupo pessoal $statusText com sucesso!");
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $grupoPessoal = GrupoPessoais::findOrFail($id);

        // Verificar se o grupo pessoal está sendo usado em algum funcionário
        // Aqui você pode adicionar lógica para verificar relações, se necessário

        $grupoPessoal->delete();

        return redirect()->route('grupo-pessoais.index')
            ->with('success', 'Grupo pessoal excluído com sucesso!');
    }
}
