<?php

namespace App\Http\Controllers;

use App\Models\Funcoes;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class FuncoesController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        // Parâmetros de filtro
        $nomeFuncao = $request->input('nomeFuncao');
        $estado = $request->input('estado');

        // Query base
        $query = Funcoes::query();

        // Aplicar filtros se fornecidos
        if ($nomeFuncao) {
            $query->where('nomeFuncao', 'like', '%' . $nomeFuncao . '%');
        }

        if ($estado !== null && $estado !== '') {
            $query->where('estado', $estado);
        }

        // Ordenar e paginar resultados
        $funcoes = $query->orderBy('id', 'desc')
                      ->paginate(10)
                      ->withQueryString();

        return view('Administracao.Funcoes', compact('funcoes'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('Administracao.CriarFuncoes');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'nomeFuncao' => 'required|string|max:255|unique:funcoes,nomeFuncao',
            'ObservacaoFuncao' => 'nullable|string|max:1000',
        ], [
            'nomeFuncao.required' => 'O nome da função é obrigatório',
            'nomeFuncao.max' => 'O nome da função não pode ter mais de 255 caracteres',
            'nomeFuncao.unique' => 'Esta função já existe',
            'ObservacaoFuncao.max' => 'A observação não pode ter mais de 1000 caracteres'
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // Criar a função
        Funcoes::create([
            'nomeFuncao' => $request->nomeFuncao,
            'ObservacaoFuncao' => $request->ObservacaoFuncao,
            'estado' => 'Ativo' // Ativo por padrão
        ]);

        return redirect()->route('funcoes.index')
            ->with('success', 'Função criada com sucesso!');
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        $funcao = Funcoes::findOrFail($id);
        return view('Administracao.DetalheFuncoes', compact('funcao'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        $funcao = Funcoes::findOrFail($id);
        return view('Administracao.EditarFuncoes', compact('funcao'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $funcao = Funcoes::findOrFail($id);

        // Validação
        $validator = Validator::make($request->all(), [
            'nomeFuncao' => 'required|string|max:255|unique:funcoes,nomeFuncao,' . $id,
            'ObservacaoFuncao' => 'nullable|string|max:1000',
            'estado' => 'required|in:Ativo,Inativo',
        ], [
            'nomeFuncao.required' => 'O nome da função é obrigatório',
            'nomeFuncao.max' => 'O nome da função não pode ter mais de 255 caracteres',
            'nomeFuncao.unique' => 'Esta função já existe',
            'ObservacaoFuncao.max' => 'A observação não pode ter mais de 1000 caracteres',
            'estado.required' => 'O estado é obrigatório',
            'estado.in' => 'Estado deve ser Ativo ou Inativo'
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // Atualizar a função
        $funcao->nomeFuncao = $request->nomeFuncao;
        $funcao->ObservacaoFuncao = $request->ObservacaoFuncao;
        $funcao->estado = $request->estado;
        $funcao->save();

        return redirect()->route('funcoes.index')
            ->with('success', 'Função atualizada com sucesso!');
    }

    /**
     * Toggle the status of the specified resource.
     */
    public function toggleStatus($id)
    {
        $funcao = Funcoes::findOrFail($id);

        // Alternar estado (Ativo <-> Inativo)
        $funcao->estado = ($funcao->estado == 'Ativo') ? 'Inativo' : 'Ativo';
        $funcao->save();

        $statusText = $funcao->estado == 'Ativo' ? 'ativada' : 'desativada';

        return redirect()->route('funcoes.index')
            ->with('success', "Função $statusText com sucesso!");
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $funcao = Funcoes::findOrFail($id);

        // Verificar se a função está sendo usada em algum funcionário
        // Aqui você pode adicionar lógica para verificar relações, se necessário

        $funcao->delete();

        return redirect()->route('funcoes.index')
            ->with('success', 'Função excluída com sucesso!');
    }
}
