<?php

namespace App\Http\Controllers;

use App\Models\Funcionario;
use App\Models\Departamento;
use App\Models\Funcoes;
use App\Models\GrupoPessoais;
use App\Models\Remuneracao;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class FuncionarioController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        // Parâmetros de filtro
        $nomeCompleto = $request->input('nomeCompleto');
        $departamento = $request->input('departamento');
        $funcao = $request->input('funcao');
        $estado = $request->input('estado');

        // Query base com relacionamentos
        $query = Funcionario::with(['departamento', 'funcao', 'grupoPessoal']);

        // Aplicar filtros se fornecidos
        if ($nomeCompleto) {
            $query->where('nomeCompleto', 'like', '%' . $nomeCompleto . '%');
        }

        if ($departamento) {
            $query->where('id_departamento', $departamento);
        }

        if ($funcao) {
            $query->where('id_funcao', $funcao);
        }

        if ($estado !== null && $estado !== '') {
            $query->where('estado', $estado);
        }

        // Ordenar e paginar resultados
        $funcionarios = $query->orderBy('id', 'desc')
                      ->paginate(10)
                      ->withQueryString();

        // Buscar dados para os filtros dropdown
        $departamentos = Departamento::ativos()->orderBy('nomeDepartamento')->get();
        $funcoes = Funcoes::ativos()->orderBy('nomeFuncao')->get();
        $grupoPessoais = GrupoPessoais::ativos()->orderBy('nomeGrupo')->get();

        // Remunerações ativas para o modal de atribuição
        $remuneracoes = Remuneracao::ativos()->orderBy('tipo')->get();

        return view('Administracao.Funcionario', compact('funcionarios', 'departamentos', 'funcoes', 'grupoPessoais', 'remuneracoes'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $departamentos = Departamento::ativos()->orderBy('nomeDepartamento')->get();
        $funcoes = Funcoes::ativos()->orderBy('nomeFuncao')->get();
        $grupoPessoais = GrupoPessoais::ativos()->orderBy('nomeGrupo')->get();

        return view('Administracao.CriarFuncionario', compact('departamentos', 'funcoes', 'grupoPessoais'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // Validação
        $validator = Validator::make($request->all(), [
            'nomeCompleto' => 'required|string|max:255',
            'numeroDocumento' => 'required|string|max:50|unique:funcionarios,numeroDocumento',
            'tipoDocumento' => 'required|string|max:50',
            'checkDocumento' => 'required|in:0,1',
            'telefone' => 'required|string|max:20|unique:funcionarios,telefone',
            'checkTelefone' => 'required|in:0,1',
            'email' => 'required|email|max:255|unique:funcionarios,email',
            'numeroColaborador' => 'nullable|string|max:50|unique:funcionarios,numeroColaborador',
            'id_departamento' => 'required|exists:departamentos,id',
            'id_funcao' => 'required|exists:funcoes,id',
            'id_grupo_pessoal' => 'required|exists:grupo_pessoais,id',
        ], [
            'nomeCompleto.required' => 'O nome completo é obrigatório',
            'numeroDocumento.required' => 'O número do documento é obrigatório',
            'numeroDocumento.unique' => 'Este número de documento já está cadastrado',
            'tipoDocumento.required' => 'O tipo de documento é obrigatório',
            'telefone.required' => 'O telefone é obrigatório',
            'telefone.unique' => 'Este telefone já está cadastrado',
            'email.required' => 'O email é obrigatório',
            'email.email' => 'O email deve ter um formato válido',
            'email.unique' => 'Este email já está cadastrado',
            'numeroColaborador.unique' => 'Este número de colaborador já está cadastrado',
            'id_departamento.required' => 'O departamento é obrigatório',
            'id_departamento.exists' => 'Departamento selecionado não existe',
            'id_funcao.required' => 'A função é obrigatória',
            'id_funcao.exists' => 'Função selecionada não existe',
            'id_grupo_pessoal.required' => 'O grupo pessoal é obrigatório',
            'id_grupo_pessoal.exists' => 'Grupo pessoal selecionado não existe',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // Criar o funcionário
        Funcionario::create([
            'nomeCompleto' => $request->nomeCompleto,
            'numeroDocumento' => $request->numeroDocumento,
            'tipoDocumento' => $request->tipoDocumento,
            'checkDocumento' => $request->checkDocumento,
            'telefone' => $request->telefone,
            'checkTelefone' => $request->checkTelefone,
            'email' => $request->email,
            'numeroColaborador' => $request->numeroColaborador,
            'estado' => 'Ativo',
            'id_departamento' => $request->id_departamento,
            'id_funcao' => $request->id_funcao,
            'id_grupo_pessoal' => $request->id_grupo_pessoal,
        ]);

        return redirect()->route('funcionarios.index')
            ->with('success', 'Funcionário criado com sucesso!');
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        $funcionario = Funcionario::with(['departamento', 'funcao', 'grupoPessoal'])->findOrFail($id);
        return view('Administracao.DetalheFuncionario', compact('funcionario'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        $funcionario = Funcionario::findOrFail($id);
        $departamentos = Departamento::ativos()->orderBy('nomeDepartamento')->get();
        $funcoes = Funcoes::ativos()->orderBy('nomeFuncao')->get();
        $grupoPessoais = GrupoPessoais::ativos()->orderBy('nomeGrupo')->get();

        return view('Administracao.EditarFuncionario', compact('funcionario', 'departamentos', 'funcoes', 'grupoPessoais'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $funcionario = Funcionario::findOrFail($id);

        // Validação
        $validator = Validator::make($request->all(), [
            'nomeCompleto' => 'required|string|max:255',
            'numeroDocumento' => 'required|string|max:50|unique:funcionarios,numeroDocumento,' . $id,
            'tipoDocumento' => 'required|string|max:50',
            'checkDocumento' => 'required|in:0,1',
            'telefone' => 'required|string|max:20|unique:funcionarios,telefone,' . $id,
            'checkTelefone' => 'required|in:0,1',
            'email' => 'required|email|max:255|unique:funcionarios,email,' . $id,
            'numeroColaborador' => 'nullable|string|max:50|unique:funcionarios,numeroColaborador,' . $id,
            'estado' => 'required|in:Ativo,Inativo',
            'id_departamento' => 'required|exists:departamentos,id',
            'id_funcao' => 'required|exists:funcoes,id',
            'id_grupo_pessoal' => 'required|exists:grupo_pessoais,id',
        ], [
            'nomeCompleto.required' => 'O nome completo é obrigatório',
            'numeroDocumento.required' => 'O número do documento é obrigatório',
            'numeroDocumento.unique' => 'Este número de documento já está cadastrado',
            'tipoDocumento.required' => 'O tipo de documento é obrigatório',
            'telefone.required' => 'O telefone é obrigatório',
            'telefone.unique' => 'Este telefone já está cadastrado',
            'email.required' => 'O email é obrigatório',
            'email.email' => 'O email deve ter um formato válido',
            'email.unique' => 'Este email já está cadastrado',
            'numeroColaborador.unique' => 'Este número de colaborador já está cadastrado',
            'estado.required' => 'O estado é obrigatório',
            'estado.in' => 'Estado deve ser Ativo ou Inativo',
            'id_departamento.required' => 'O departamento é obrigatório',
            'id_departamento.exists' => 'Departamento selecionado não existe',
            'id_funcao.required' => 'A função é obrigatória',
            'id_funcao.exists' => 'Função selecionada não existe',
            'id_grupo_pessoal.required' => 'O grupo pessoal é obrigatório',
            'id_grupo_pessoal.exists' => 'Grupo pessoal selecionado não existe',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // Atualizar o funcionário
        $funcionario->nomeCompleto = $request->nomeCompleto;
        $funcionario->numeroDocumento = $request->numeroDocumento;
        $funcionario->tipoDocumento = $request->tipoDocumento;
        $funcionario->checkDocumento = $request->checkDocumento;
        $funcionario->telefone = $request->telefone;
        $funcionario->checkTelefone = $request->checkTelefone;
        $funcionario->email = $request->email;
        $funcionario->numeroColaborador = $request->numeroColaborador;
        $funcionario->estado = $request->estado;
        $funcionario->id_departamento = $request->id_departamento;
        $funcionario->id_funcao = $request->id_funcao;
        $funcionario->id_grupo_pessoal = $request->id_grupo_pessoal;
        $funcionario->save();

        return redirect()->route('funcionarios.index')
            ->with('success', 'Funcionário atualizado com sucesso!');
    }

    /**
     * Toggle the status of the specified resource.
     */
    public function toggleStatus($id)
    {
        $funcionario = Funcionario::findOrFail($id);

        // Alternar estado (Ativo <-> Inativo)
        $funcionario->estado = ($funcionario->estado == 'Ativo') ? 'Inativo' : 'Ativo';
        $funcionario->save();

        $statusText = $funcionario->estado == 'Ativo' ? 'ativado' : 'desativado';

        return redirect()->route('funcionarios.index')
            ->with('success', "Funcionário $statusText com sucesso!");
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $funcionario = Funcionario::findOrFail($id);

        $funcionario->delete();

        return redirect()->route('funcionarios.index')
            ->with('success', 'Funcionário excluído com sucesso!');
    }

    /**
     * Atribui uma remuneração a um funcionário (cria/atualiza dados no pivot)
     */
    public function storeRemuneracao(Request $request, $id)
    {
        $funcionario = Funcionario::findOrFail($id);

        $data = $request->validate([
            'remuneracao_id' => 'required|exists:remuneracoes,id',
            'valor' => 'required|numeric|min:0',
            'status' => 'required|in:Ativo,Inativo',
            'observacao' => 'nullable|string|max:1000',
        ]);

        // Se já existir, atualiza; senão, cria sem remover as outras
        $funcionario->remuneracoes()->syncWithoutDetaching([
            $data['remuneracao_id'] => [
                'valor' => $data['valor'],
                'status' => $data['status'],
                'observacao' => $data['observacao'] ?? null,
            ]
        ]);

        // Caso já exista, garantimos update no pivot
        $funcionario->remuneracoes()->updateExistingPivot($data['remuneracao_id'], [
            'valor' => $data['valor'],
            'status' => $data['status'],
            'observacao' => $data['observacao'] ?? null,
        ]);

        return redirect()->route('funcionarios.index')
            ->with('success', 'Remuneração atribuída ao funcionário com sucesso!');
    }

    /**
     * Obter coordenadas bancárias do funcionário (AJAX)
     */
    public function getCoordenadas($id)
    {
        $funcionario = Funcionario::findOrFail($id);
        
        return response()->json([
            'banco' => $funcionario->banco,
            'iban' => $funcionario->iban,
            'nib' => $funcionario->nib,
            'swift_bic' => $funcionario->swift_bic,
            'observacoes_bancarias' => $funcionario->observacoes_bancarias
        ]);
    }

    /**
     * Atualizar coordenadas bancárias do funcionário
     */
    public function updateCoordenadas(Request $request, $id)
    {
        $funcionario = Funcionario::findOrFail($id);

        // Validação
        $validated = $request->validate([
            'banco' => 'nullable|string|max:255',
            'iban' => 'nullable|string|max:50',
            'nib' => 'nullable|string|max:50',
            'swift_bic' => 'nullable|string|max:20',
            'titular_conta' => 'nullable|string|max:255',
            'observacoes_bancarias' => 'nullable|string|max:1000'
        ]);

        // Atualizar
        $funcionario->update($validated);

        return redirect()->route('funcionarios.index')
            ->with('success', 'Coordenadas bancárias atualizadas com sucesso!');
    }
}
