<?php

namespace App\Http\Controllers;

use App\Models\Projecto;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;

class DesembolsoController extends Controller
{
    // Listagem de Desembolso
    public function indexAdmin()
    {
        $projetos = Projecto::with('tecnicoProjecto')
            ->orderBy('tecnico_projectoFK')
            ->get()
            ->groupBy(function ($projeto) {
                return $projeto->tecnicoProjecto ? $projeto->tecnicoProjecto->name : 'Sem Técnico';
            });

        // Cálculo dos totais
        $totalPago = Projecto::with('pagamentos')->get()->sum(function ($projeto) {
            return $projeto->valorTotalPago();
        });

        $totalProjetosExecucao = Projecto::where('status', ['4', '5'])->sum('limiteorcamentalP');

        return view('Administracao.desembolso', compact('projetos', 'totalPago', 'totalProjetosExecucao'));
    }

    /**
     * Exportação geral em PDF
     */
    public function exportarPDFGeral()
    {
        $projetos = Projecto::with('tecnicoProjecto', 'tipoProjecto')
            ->orderBy('tecnico_projectoFK')
            ->get()
            ->groupBy(function ($projeto) {
                return $projeto->tecnicoProjecto ? $projeto->tecnicoProjecto->name : 'Sem Técnico';
            });

        // Cálculo dos totais
        $totalPago = Projecto::with('pagamentos')->get()->sum(function ($projeto) {
            return $projeto->valorTotalPago();
        });

        $totalProjetosExecucao = Projecto::where('status', ['4', '5'])->sum('limiteorcamentalP');

        $dados = [
            'projetos' => $projetos,
            'totalPago' => $totalPago,
            'totalProjetosExecucao' => $totalProjetosExecucao,
            'dataGeracao' => now()->format('d/m/Y H:i:s')
        ];

        $pdf = Pdf::loadView('pdf.desembolso-geral', $dados)
                  ->setPaper('A4', 'landscape');
                  
        return $pdf->download('Relatorio_Desembolsos_Geral_' . now()->format('Y-m-d_H-i-s') . '.pdf');
    }

    /**
     * Visualização do PDF geral
     */
    public function visualizarPDFGeral()
    {
        $projetos = Projecto::with('tecnicoProjecto', 'tipoProjecto')
            ->orderBy('tecnico_projectoFK')
            ->get()
            ->groupBy(function ($projeto) {
                return $projeto->tecnicoProjecto ? $projeto->tecnicoProjecto->name : 'Sem Técnico';
            });

        $totalPago = Projecto::with('pagamentos')->get()->sum(function ($projeto) {
            return $projeto->valorTotalPago();
        });

        $totalProjetosExecucao = Projecto::where('status', ['4', '5'])->sum('limiteorcamentalP');

        $dados = [
            'projetos' => $projetos,
            'totalPago' => $totalPago,
            'totalProjetosExecucao' => $totalProjetosExecucao,
            'dataGeracao' => now()->format('d/m/Y H:i:s')
        ];

        $pdf = Pdf::loadView('pdf.desembolso-geral', $dados)
                  ->setPaper('A4', 'landscape');
                  
        return $pdf->stream('Relatorio_Desembolsos_Geral.pdf');
    }

    /**
     * Exportação geral em Excel
     */
    public function exportarExcelGeral()
    {
        $projetos = Projecto::with('tecnicoProjecto', 'tipoProjecto')
            ->orderBy('tecnico_projectoFK')
            ->get()
            ->groupBy(function ($projeto) {
                return $projeto->tecnicoProjecto ? $projeto->tecnicoProjecto->name : 'Sem Técnico';
            });

        $totalPago = Projecto::with('pagamentos')->get()->sum(function ($projeto) {
            return $projeto->valorTotalPago();
        });

        $totalProjetosExecucao = Projecto::where('status', ['4', '5'])->sum('limiteorcamentalP');

        $filename = 'Relatorio_Desembolsos_Geral_' . now()->format('Y-m-d_H-i-s') . '.csv';
        
        $headers = array(
            "Content-type" => "text/csv; charset=UTF-8",
            "Content-Disposition" => "attachment; filename=$filename",
            "Pragma" => "no-cache",
            "Cache-Control" => "must-revalidate, post-check=0, pre-check=0",
            "Expires" => "0"
        );

        $callback = function() use ($projetos, $totalPago, $totalProjetosExecucao) {
            $file = fopen('php://output', 'w');
            
            // Adicionar BOM para UTF-8
            fputs($file, "\xEF\xBB\xBF");
            
            // Cabeçalho do relatório
            fputcsv($file, ['RELATÓRIO DE DESEMBOLSOS - GERAL'], ';');
            fputcsv($file, ['Data de Geração: ' . now()->format('d/m/Y H:i:s')], ';');
            fputcsv($file, [''], ';'); // Linha em branco
            
            // Totais gerais
            fputcsv($file, ['RESUMO GERAL'], ';');
            fputcsv($file, ['Total Orçamentado', 'Total Desembolsado', 'Total por Desembolsar'], ';');
            fputcsv($file, [
                number_format($totalProjetosExecucao, 2, ',', '.') . ' Kz',
                number_format($totalPago, 2, ',', '.') . ' Kz',
                number_format($totalProjetosExecucao - $totalPago, 2, ',', '.') . ' Kz'
            ], ';');
            fputcsv($file, [''], ';'); // Linha em branco
            
            // Cabeçalho dos projetos
            fputcsv($file, [
                'Instituição',
                'Código Projeto',
                'Nome do Projeto',
                'Ano Início',
                'Ano Fim',
                'Duração (Anos)',
                'Status',
                'Província',
                'Localização',
                'Tipo de Projeto',
                'Custo Total (Kz)',
                'Valor Desembolsado (Kz)',
                'Execução Financeira (%)',
                'Valor por Desembolsar (Kz)'
            ], ';');
            
            // Dados dos projetos
            foreach ($projetos as $instituicao => $projetosInstituicao) {
                foreach ($projetosInstituicao as $projeto) {
                    $execucao = $projeto->limiteorcamentalP > 0 ? 
                        ($projeto->valorTotalPago() / $projeto->limiteorcamentalP) * 100 : 0;
                    
                    fputcsv($file, [
                        $instituicao ?? 'N/A',
                        $projeto->codProjecto,
                        $projeto->nomeProjecto,
                        date('Y', strtotime($projeto->duracaoP)),
                        date('Y', strtotime($projeto->duracaoF)),
                        (date('Y', strtotime($projeto->duracaoF)) - date('Y', strtotime($projeto->duracaoP))),
                        ['Recebido', 'Em Análise', 'Aprovado', 'Rejeitado', 'Em Execução', 'Concluído'][$projeto->status] ?? 'N/A',
                        $projeto->provincia ?? 'N/A',
                        $projeto->localizacao_projecto ?? 'N/A',
                        $projeto->tipoProjecto->descricao ?? 'N/A',
                        number_format($projeto->limiteorcamentalP, 2, ',', '.'),
                        number_format($projeto->valorTotalPago(), 2, ',', '.'),
                        number_format($execucao, 2, ',', '.') . '%',
                        number_format($projeto->limiteorcamentalP - $projeto->valorTotalPago(), 2, ',', '.')
                    ], ';');
                }
            }
            
            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    /**
     * Exportação por instituição em PDF
     */
    public function exportarPDFInstituicao(Request $request)
    {
        $instituicao = $request->get('instituicao');
        
        $projetos = Projecto::with('tecnicoProjecto', 'tipoProjecto')
            ->whereHas('tecnicoProjecto', function($query) use ($instituicao) {
                $query->where('name', $instituicao);
            })
            ->orderBy('codProjecto')
            ->get();

        if ($projetos->isEmpty()) {
            return redirect()->back()->with('error', 'Nenhum projeto encontrado para esta instituição.');
        }

        $totalCusto = $projetos->sum('limiteorcamentalP');
        $totalDesembolsado = $projetos->sum(function ($projeto) {
            return $projeto->valorTotalPago();
        });

        $dados = [
            'projetos' => $projetos,
            'instituicao' => $instituicao,
            'totalCusto' => $totalCusto,
            'totalDesembolsado' => $totalDesembolsado,
            'dataGeracao' => now()->format('d/m/Y H:i:s')
        ];

        $pdf = Pdf::loadView('pdf.desembolso-instituicao', $dados)
                  ->setPaper('A4', 'landscape');
                  
        return $pdf->download('Desembolsos_' . str_replace(' ', '_', $instituicao) . '_' . now()->format('Y-m-d') . '.pdf');
    }

    /**
     * Visualização PDF por instituição
     */
    public function visualizarPDFInstituicao(Request $request)
    {
        $instituicao = $request->get('instituicao');
        
        $projetos = Projecto::with('tecnicoProjecto', 'tipoProjecto')
            ->whereHas('tecnicoProjecto', function($query) use ($instituicao) {
                $query->where('name', $instituicao);
            })
            ->orderBy('codProjecto')
            ->get();

        $totalCusto = $projetos->sum('limiteorcamentalP');
        $totalDesembolsado = $projetos->sum(function ($projeto) {
            return $projeto->valorTotalPago();
        });

        $dados = [
            'projetos' => $projetos,
            'instituicao' => $instituicao,
            'totalCusto' => $totalCusto,
            'totalDesembolsado' => $totalDesembolsado,
            'dataGeracao' => now()->format('d/m/Y H:i:s')
        ];

        $pdf = Pdf::loadView('pdf.desembolso-instituicao', $dados)
                  ->setPaper('A4', 'landscape');
                  
        return $pdf->stream('Desembolsos_' . str_replace(' ', '_', $instituicao) . '.pdf');
    }

    /**
     * Exportação por instituição em Excel
     */
    public function exportarExcelInstituicao(Request $request)
    {
        $instituicao = $request->get('instituicao');
        
        $projetos = Projecto::with('tecnicoProjecto', 'tipoProjecto')
            ->whereHas('tecnicoProjecto', function($query) use ($instituicao) {
                $query->where('name', $instituicao);
            })
            ->orderBy('codProjecto')
            ->get();

        if ($projetos->isEmpty()) {
            return redirect()->back()->with('error', 'Nenhum projeto encontrado para esta instituição.');
        }

        $totalCusto = $projetos->sum('limiteorcamentalP');
        $totalDesembolsado = $projetos->sum(function ($projeto) {
            return $projeto->valorTotalPago();
        });

        $filename = 'Desembolsos_' . str_replace(' ', '_', $instituicao) . '_' . now()->format('Y-m-d_H-i-s') . '.csv';
        
        $headers = array(
            "Content-type" => "text/csv; charset=UTF-8",
            "Content-Disposition" => "attachment; filename=$filename",
            "Pragma" => "no-cache",
            "Cache-Control" => "must-revalidate, post-check=0, pre-check=0",
            "Expires" => "0"
        );

        $callback = function() use ($projetos, $instituicao, $totalCusto, $totalDesembolsado) {
            $file = fopen('php://output', 'w');
            
            // Adicionar BOM para UTF-8
            fputs($file, "\xEF\xBB\xBF");
            
            // Cabeçalho do relatório
            fputcsv($file, ['RELATÓRIO DE DESEMBOLSOS - ' . strtoupper($instituicao)], ';');
            fputcsv($file, ['Data de Geração: ' . now()->format('d/m/Y H:i:s')], ';');
            fputcsv($file, [''], ';'); // Linha em branco
            
            // Totais da instituição
            fputcsv($file, ['RESUMO DA INSTITUIÇÃO'], ';');
            fputcsv($file, ['Total Orçamentado', 'Total Desembolsado', 'Total por Desembolsar', 'Percentual Executado'], ';');
            $percentual = $totalCusto > 0 ? ($totalDesembolsado / $totalCusto) * 100 : 0;
            fputcsv($file, [
                number_format($totalCusto, 2, ',', '.') . ' Kz',
                number_format($totalDesembolsado, 2, ',', '.') . ' Kz',
                number_format($totalCusto - $totalDesembolsado, 2, ',', '.') . ' Kz',
                number_format($percentual, 1, ',', '.') . '%'
            ], ';');
            fputcsv($file, [''], ';'); // Linha em branco
            
            // Cabeçalho dos projetos
            fputcsv($file, [
                'Código Projeto',
                'Nome do Projeto',
                'Ano Início',
                'Ano Fim',
                'Duração (Anos)',
                'Status',
                'Província',
                'Localização',
                'Tipo de Projeto',
                'Custo Total (Kz)',
                'Valor Desembolsado (Kz)',
                'Execução Financeira (%)',
                'Valor por Desembolsar (Kz)'
            ], ';');
            
            // Dados dos projetos
            foreach ($projetos as $projeto) {
                $execucao = $projeto->limiteorcamentalP > 0 ? 
                    ($projeto->valorTotalPago() / $projeto->limiteorcamentalP) * 100 : 0;
                
                fputcsv($file, [
                    $projeto->codProjecto,
                    $projeto->nomeProjecto,
                    date('Y', strtotime($projeto->duracaoP)),
                    date('Y', strtotime($projeto->duracaoF)),
                    (date('Y', strtotime($projeto->duracaoF)) - date('Y', strtotime($projeto->duracaoP))),
                    ['Recebido', 'Em Análise', 'Aprovado', 'Rejeitado', 'Em Execução', 'Concluído'][$projeto->status] ?? 'N/A',
                    $projeto->provincia ?? 'N/A',
                    $projeto->localizacao_projecto ?? 'N/A',
                    $projeto->tipoProjecto->descricao ?? 'N/A',
                    number_format($projeto->limiteorcamentalP, 2, ',', '.'),
                    number_format($projeto->valorTotalPago(), 2, ',', '.'),
                    number_format($execucao, 2, ',', '.') . '%',
                    number_format($projeto->limiteorcamentalP - $projeto->valorTotalPago(), 2, ',', '.')
                ], ';');
            }
            
            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}
