<?php

namespace App\Http\Controllers;

use App\Models\Departamento;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class DepartamentoController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        // Parâmetros de filtro
        $nomeDepartamento = $request->input('nomeDepartamento');
        $estado = $request->input('estado');

        // Query base
        $query = Departamento::query();

        // Aplicar filtros se fornecidos
        if ($nomeDepartamento) {
            $query->where('nomeDepartamento', 'like', '%' . $nomeDepartamento . '%');
        }

        if ($estado !== null && $estado !== '') {
            $query->where('estado', $estado);
        }

        // Ordenar e paginar resultados
        $departamentos = $query->orderBy('id', 'desc')
                      ->paginate(10)
                      ->withQueryString();

        return view('Administracao.Departamento', compact('departamentos'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('Administracao.CriarDepartamento');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // Validação
        $validator = Validator::make($request->all(), [
            'nomeDepartamento' => 'required|string|max:255|unique:departamentos,nomeDepartamento',
            'ObservacaoDepartamento' => 'nullable|string|max:1000',
        ], [
            'nomeDepartamento.required' => 'O nome do departamento é obrigatório',
            'nomeDepartamento.max' => 'O nome do departamento não pode ter mais de 255 caracteres',
            'nomeDepartamento.unique' => 'Este departamento já existe',
            'ObservacaoDepartamento.max' => 'A observação não pode ter mais de 1000 caracteres'
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // Criar o departamento
        Departamento::create([
            'nomeDepartamento' => $request->nomeDepartamento,
            'ObservacaoDepartamento' => $request->ObservacaoDepartamento,
            'estado' => 'Ativo' // Ativo por padrão
        ]);

        return redirect()->route('departamentos.index')
            ->with('success', 'Departamento criado com sucesso!');
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        $departamento = Departamento::findOrFail($id);
        return view('Administracao.DetalheDepartamento', compact('departamento'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        $departamento = Departamento::findOrFail($id);
        return view('Administracao.EditarDepartamento', compact('departamento'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $departamento = Departamento::findOrFail($id);

        // Validação
        $validator = Validator::make($request->all(), [
            'nomeDepartamento' => 'required|string|max:255|unique:departamentos,nomeDepartamento,' . $id,
            'ObservacaoDepartamento' => 'nullable|string|max:1000',
            'estado' => 'required|in:Ativo,Inativo',
        ], [
            'nomeDepartamento.required' => 'O nome do departamento é obrigatório',
            'nomeDepartamento.max' => 'O nome do departamento não pode ter mais de 255 caracteres',
            'nomeDepartamento.unique' => 'Este departamento já existe',
            'ObservacaoDepartamento.max' => 'A observação não pode ter mais de 1000 caracteres',
            'estado.required' => 'O estado é obrigatório',
            'estado.in' => 'Estado deve ser Ativo ou Inativo'
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // Atualizar o departamento
        $departamento->nomeDepartamento = $request->nomeDepartamento;
        $departamento->ObservacaoDepartamento = $request->ObservacaoDepartamento;
        $departamento->estado = $request->estado;
        $departamento->save();

        return redirect()->route('departamentos.index')
            ->with('success', 'Departamento atualizado com sucesso!');
    }

    /**
     * Toggle the status of the specified resource.
     */
    public function toggleStatus($id)
    {
        $departamento = Departamento::findOrFail($id);

        // Alternar estado (Ativo <-> Inativo)
        $departamento->estado = ($departamento->estado == 'Ativo') ? 'Inativo' : 'Ativo';
        $departamento->save();

        $statusText = $departamento->estado == 'Ativo' ? 'ativado' : 'desativado';

        return redirect()->route('departamentos.index')
            ->with('success', "Departamento $statusText com sucesso!");
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $departamento = Departamento::findOrFail($id);

        // Verificar se o departamento está sendo usado em algum funcionário
        // Aqui você pode adicionar lógica para verificar relações, se necessário

        $departamento->delete();

        return redirect()->route('departamentos.index')
            ->with('success', 'Departamento excluído com sucesso!');
    }
}
