<?php

namespace App\Http\Controllers;

use App\Models\CoordenadasBancarias;
use App\Models\Funcionario;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class CoordenadasBancariasController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        // Parâmetros de filtro
        $nomeBanco = $request->input('nomeBanco');
        $siglaBanco = $request->input('siglaBanco');
        $titular = $request->input('titular');
        $estado = $request->input('estado');

        // Query base com relacionamentos
        $query = CoordenadasBancarias::with(['funcionario']);

        // Aplicar filtros se fornecidos
        if ($nomeBanco) {
            $query->where('nomeBanco', 'like', '%' . $nomeBanco . '%');
        }

        if ($siglaBanco) {
            $query->where('siglaBanco', 'like', '%' . $siglaBanco . '%');
        }

        if ($titular) {
            $query->where('id_titular', $titular);
        }

        if ($estado !== null && $estado !== '') {
            $query->where('estado', $estado);
        }

        // Ordenar e paginar resultados
        $coordenadasBancarias = $query->orderBy('id', 'desc')
                      ->paginate(10)
                      ->withQueryString();

        // Buscar funcionários ativos para o filtro dropdown
        $funcionarios = Funcionario::ativos()->orderBy('nomeCompleto')->get();

        return view('Administracao.CoordenadasBancarias', compact('coordenadasBancarias', 'funcionarios'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $funcionarios = Funcionario::ativos()->orderBy('nomeCompleto')->get();
        return view('Administracao.CriarCoordenadasBancarias', compact('funcionarios'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // Validação
        $validator = Validator::make($request->all(), [
            'nomeBanco' => 'required|string|max:255',
            'siglaBanco' => 'required|string|max:20|unique:coordenadas_bancarias,siglaBanco',
            'ContaBancaria' => 'required|string|max:50',
            'Iban' => 'required|string|max:34|unique:coordenadas_bancarias,Iban',
            'id_titular' => 'required|exists:funcionarios,id',
        ], [
            'nomeBanco.required' => 'O nome do banco é obrigatório',
            'siglaBanco.required' => 'A sigla do banco é obrigatória',
            'siglaBanco.unique' => 'Esta sigla de banco já está cadastrada',
            'ContaBancaria.required' => 'O número da conta bancária é obrigatório',
            'Iban.required' => 'O IBAN é obrigatório',
            'Iban.unique' => 'Este IBAN já está cadastrado',
            'id_titular.required' => 'O titular da conta é obrigatório',
            'id_titular.exists' => 'Funcionário selecionado não existe',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // Criar as coordenadas bancárias
        CoordenadasBancarias::create([
            'nomeBanco' => $request->nomeBanco,
            'siglaBanco' => $request->siglaBanco,
            'ContaBancaria' => $request->ContaBancaria,
            'Iban' => $request->Iban,
            'id_titular' => $request->id_titular,
            'estado' => 'Ativo',
        ]);

        return redirect()->route('coordenadas-bancarias.index')
            ->with('success', 'Coordenadas bancárias criadas com sucesso!');
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        $coordenadasBancarias = CoordenadasBancarias::with(['funcionario'])->findOrFail($id);
        return view('Administracao.DetalheCoordenadasBancarias', compact('coordenadasBancarias'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        $coordenadasBancarias = CoordenadasBancarias::findOrFail($id);
        $funcionarios = Funcionario::ativos()->orderBy('nomeCompleto')->get();

        return view('Administracao.EditarCoordenadasBancarias', compact('coordenadasBancarias', 'funcionarios'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $coordenadasBancarias = CoordenadasBancarias::findOrFail($id);

        // Validação
        $validator = Validator::make($request->all(), [
            'nomeBanco' => 'required|string|max:255',
            'siglaBanco' => 'required|string|max:20|unique:coordenadas_bancarias,siglaBanco,' . $id,
            'ContaBancaria' => 'required|string|max:50',
            'Iban' => 'required|string|max:34|unique:coordenadas_bancarias,Iban,' . $id,
            'estado' => 'required|in:Ativo,Inativo',
            'id_titular' => 'required|exists:funcionarios,id',
        ], [
            'nomeBanco.required' => 'O nome do banco é obrigatório',
            'siglaBanco.required' => 'A sigla do banco é obrigatória',
            'siglaBanco.unique' => 'Esta sigla de banco já está cadastrada',
            'ContaBancaria.required' => 'O número da conta bancária é obrigatório',
            'Iban.required' => 'O IBAN é obrigatório',
            'Iban.unique' => 'Este IBAN já está cadastrado',
            'estado.required' => 'O estado é obrigatório',
            'estado.in' => 'Estado deve ser Ativo ou Inativo',
            'id_titular.required' => 'O titular da conta é obrigatório',
            'id_titular.exists' => 'Funcionário selecionado não existe',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // Atualizar as coordenadas bancárias
        $coordenadasBancarias->nomeBanco = $request->nomeBanco;
        $coordenadasBancarias->siglaBanco = $request->siglaBanco;
        $coordenadasBancarias->ContaBancaria = $request->ContaBancaria;
        $coordenadasBancarias->Iban = $request->Iban;
        $coordenadasBancarias->estado = $request->estado;
        $coordenadasBancarias->id_titular = $request->id_titular;
        $coordenadasBancarias->save();

        return redirect()->route('coordenadas-bancarias.index')
            ->with('success', 'Coordenadas bancárias atualizadas com sucesso!');
    }

    /**
     * Toggle the status of the specified resource.
     */
    public function toggleStatus($id)
    {
        $coordenadasBancarias = CoordenadasBancarias::findOrFail($id);

        // Alternar estado (Ativo <-> Inativo)
        $coordenadasBancarias->estado = ($coordenadasBancarias->estado == 'Ativo') ? 'Inativo' : 'Ativo';
        $coordenadasBancarias->save();

        $statusText = $coordenadasBancarias->estado == 'Ativo' ? 'ativadas' : 'desativadas';

        return redirect()->route('coordenadas-bancarias.index')
            ->with('success', "Coordenadas bancárias $statusText com sucesso!");
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $coordenadasBancarias = CoordenadasBancarias::findOrFail($id);

        $coordenadasBancarias->delete();

        return redirect()->route('coordenadas-bancarias.index')
            ->with('success', 'Coordenadas bancárias excluídas com sucesso!');
    }
}
