<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\RelatorioExecucao;
use App\Models\User;
use Illuminate\Support\Facades\DB;

class CorrigirRelatoriosUsuarioNulo extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'relatorios:corrigir-usuario-nulo';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Corrige relatórios de execução com usuario_id nulo';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Iniciando correção de relatórios com usuario_id nulo...');

        try {
            // Encontrar relatórios com usuario_id nulo
            $relatoriosNulos = RelatorioExecucao::whereNull('usuario_id')->get();
            
            if ($relatoriosNulos->isEmpty()) {
                $this->info('✅ Nenhum relatório com usuario_id nulo encontrado.');
                return Command::SUCCESS;
            }

            $this->warn("Encontrados {$relatoriosNulos->count()} relatórios com usuario_id nulo.");

            // Tentar encontrar um usuário padrão (primeiro admin ou primeiro usuário)
            $usuarioPadrao = User::where('admin', 1)->first() ?? User::first();

            if (!$usuarioPadrao) {
                $this->error('❌ Nenhum usuário encontrado para definir como padrão.');
                return Command::FAILURE;
            }

            $this->info("Usando usuário padrão: {$usuarioPadrao->name} (ID: {$usuarioPadrao->id})");

            // Perguntar ao usuário se deve prosseguir
            if (!$this->confirm('Deseja prosseguir com a correção?')) {
                $this->info('Operação cancelada.');
                return Command::SUCCESS;
            }

            // Atualizar os relatórios
            DB::beginTransaction();

            $corrigidos = 0;
            foreach ($relatoriosNulos as $relatorio) {
                $relatorio->update(['usuario_id' => $usuarioPadrao->id]);
                $corrigidos++;
                
                $this->info("Relatório ID {$relatorio->id} corrigido.");
            }

            DB::commit();

            $this->info("✅ {$corrigidos} relatórios corrigidos com sucesso!");

        } catch (\Exception $e) {
            DB::rollBack();
            $this->error("❌ Erro durante a correção: " . $e->getMessage());
            return Command::FAILURE;
        }

        return Command::SUCCESS;
    }
}
